# GEOG 375 Spring 2020 Final Project

# Created by:  Benjamin Tang tang.benjamin@gmail
# Created on:  05.16.2020
# Base code by:   Nathan Jennings

'''
This script exports a series of maps from an ArcGIS Pro project to PDF. The Pro project uses recent (May 11, 2020) COVID-19 case data by county from states in the Western States Pact (California, Oregon, Washington, Nevada and Colorado), which agreed to coordinate on reopening their economies after coronavirus lockdown.

Map enumeration units are counties in two map layers:
1) 'counties_cases_West', which joins spatial county census data with numeric COVID case data and symbolized as choropleth (cases normalized to county population)
2) 'counties_cases_limited', which is a copy of counties_cases_West but symbolized by dot density and with limiting attributes (water, parkland cover) clipped away

Instead of county maps, I exported PDF maps for each entire state for broader context (the map legend is classed based on data from all five states, so exporting just a county map with a regional legend might be odd).
'''

import arcpy, os, sys, traceback, datetime

print ('GEOG 375 Spring 2020 Final Project')
# print ('Version with preset statelist') # if CHEAT below when 'counties_cases_West' unaccessible
author = 'Ben Tang'
todays_date = datetime.date.today().strftime('%d %B %Y') # European style, day before month!

# ArcGIS path variables
datapath = 'D:\\Los Rios GIS\\GEOG 375 - Intro to GIS Programming\\COVID-19_Project\\'
arcpy.env.workspace = datapath + 'COVID-19_Project.gdb'
exportpath = datapath # output PDFs to same folder as project


try:
	aprx = arcpy.mp.ArcGISProject(datapath + "COVID-19_Project.aprx")
	maplist = aprx.listMaps("Map") [0]

	# portrait layout for tall states: CA, NV
	layout_p = aprx.listLayouts("Layout_Portrait")[0]
	# landscape layout for wide states: OR, WA, CO
	layout_l = aprx.listLayouts("Layout_Landscape")[0]
	# print (layout_p.name)
	# print (layout_l.name) # check second layout correctly saved

	mapframe_p = layout_p.listElements("MAPFRAME_ELEMENT", "Map Frame")[0]
	mapframe_l = layout_l.listElements("MAPFRAME_ELEMENT", "Map Frame")[0]

	tElements_p = layout_p.listElements("TEXT_ELEMENT")
	tElements_l = layout_l.listElements("TEXT_ELEMENT")

	# Getting list of states from 'counties_cases_West' map layer
	# Initially had difficulty getting 'counties_cases_West' recognized as feature class/layer
	# Python for ArcGIS Pro online help wasn't helpful at all for such a simple task!
	fc = maplist.listLayers("counties_cases_West")[0]
	states = [row[0] for row in arcpy.da.SearchCursor(fc,'STATE_NAME')]
	statelist = set(states) # remove duplicates, as multiple counties have same STATE_NAME
	print ('\nStates to be mapped:')
	print (statelist)

	# CHEAT: preset statelist instead of reading from 'counties_cases_West' layer
	# statelist = ["California", "Nevada", "Oregon", "Washington", "Colorado"]

	# Making map for each state:
	for state in statelist:

		print ('\nCreating map of ' + state) # check correct state
		query = """"STATE_NAME" = '""" + state + """'"""
		# print (query) # check correct query

		for TOCLayer in maplist.listLayers():
		# print ("Layer Name: " + str(TOCLayer.longName)) # check layers are correct

			if TOCLayer.name == "counties_cases_West":

				# Select county features for queried state
				arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query)
				result = arcpy.GetCount_management(TOCLayer)
				print ("Selected features: " + str(result) + " counties in " + state)
				# Correct number of counties: CA 58, NV 17, OR 36, WA 39, CO 64

				# Set the proper layout orientation
				if state in ("California", "Nevada"): # tall states use portrait layout
					layout = layout_p
					mapframe = mapframe_p
					tElements = tElements_p
				else: # wide states use landscape layout
					layout = layout_l
					mapframe = mapframe_l
					tElements = tElements_l

				# Zoom to selected features then clear selection
				layer_extent = mapframe.getLayerExtent(TOCLayer)
				mapframe.camera.setExtent(layer_extent)
				mapframe.camera.scale = mapframe.camera.scale * 1.1
				arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")

				# Retitle map for each state
				for tElement in tElements:
					if tElement.name == 'Subtitle':
						tElement.text = state + ', ' + str(todays_date)

				print ('Exporting map of ' + state + '...')

				# Delete any existing copies of map, then export
				if arcpy.Exists(exportpath + 'COVID-19_' + state + '.pdf'):
					arcpy.Delete_management(exportpath + 'COVID-19_' + state + '.pdf')
				layout.exportToPDF(exportpath + "COVID-19_" + state + ".pdf")

	print ('\nCompleted Map Exports')


except:
	tb = sys.exc_info()[2]
	tbinfo = traceback.format_tb(tb)[0]
	pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" +  str(sys.exc_info()[1])
	msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

	arcpy.AddError(msgs)
	arcpy.AddError(pymsg)

	print (msgs)
	print (pymsg)

	arcpy.AddMessage(arcpy.GetMessages(1))
	print (arcpy.GetMessages(1))
