# Created by:  Ian SKinner
#              www.ilsweb.com
# Created on:   9 May 2020
# Updated on:  15 May 2020
# Copyright:   2020

'''
    - Mosaic the ~71 NED tiles out of the 111 tiles downloaded from the National
    Map Download site that overlay the WBD HU18-02 extent.
    
    - Project Mosaic raster to  NAD 1983 Geographic Coordinate System and the
    NAVD 1988 Vertical Coordinate System as ArcGIS Pro does not seem to
    automatically read the VCS from the geotiff files.
    
    - Adjust the elevation data of the Mosaic raster -200 to represent a 200
    meter sea rise for potential elimination of global glacier and polar ice.
    
    - Round the elevation values to whole integers and then COPY RASTER to
    convert the mosaic raster from 32 bit float to 16 bit signed cell precision.
    
        -- Possible enhancement would be to find a Foot based VCS and then
        convert the meter elevation to foot elevation then round to whole
        integers and COPY RASTER to 16 bit signed cell precision.
        
    - Convert all coastal cells in the Mosaic with a zero or less value to
    NoData values so they are transparent and will show data from an
    underlying NOAA ETOPO1 raster layer.  But not interior cells below zero
    elevation such as the Death Valley area.
    
'''

import os, arcpy, sys, traceback, time

def time_format(timeInSeconds):
    print(f"{int(timeInSeconds/3600):>3}H:{int((timeInSeconds%3600)/60):02d}M:{(timeInSeconds%60):04.1f}S")

try:
    processStart = time.perf_counter()
    #Init Global Variables
    ProjectPath = "P:\\Python\\Final\\"

    #Print Project Title
    print('*****************\n* GEOG 375      *\n* Final Project *\n* Ian Skinner   *\n*****************\n')

    #Set workspace to USGS Waterbasin Data Geodatbase
    arcpy.env.workspace = ProjectPath + "Data\\WBD_18_HU2_GDB.gdb\\"

    #Set path to Top Level Water Basin Feature Class
    WBDHU2 = "WBD\\WBDHU2"

    #Extract polygon geometry of the Top Level Water Basin
    for row in arcpy.da.SearchCursor(WBDHU2, ["SHAPE@", "NAME", "Value"]):
        print("WBD HU2 Feature Class")
        print("{} Value={} [N{:0.4f}:S{:0.4f}:E{:0.4f}:W{:0.4f}]\n".format(row[1],row[2],row[0].extent.YMax,row[0].extent.YMin,row[0].extent.XMax,row[0].extent.XMin))

    #Set workspace to Arcgis Pro Project directory
    arcpy.env.workspace = ProjectPath

    # Create Spatial Reference object of the desired geographic coordinate system and vertical coordinate system to set the rasters.
    # 4269: North American Datum of 1983
    # 5703: North American Vertical Datum 1988
    spatialReference = arcpy.SpatialReference(4269,5703)

    # if the Final geodatabase does not exist, create it.
    if not arcpy.Exists(ProjectPath + "Final.gdb"):
        arcpy.CreateFileGDB_management(ProjectPath,"Final.gdb")

    print("\n******************************************************************************\nLoad USGS Elevation Rasters that overlap the area of interest WBD HU2 Polygon.")
    #Ask for User Input on wether to run the create and mosaic raster functionality.  This can take up to an hour to run.    
    UIinput = ''
    
    #Keep promting the user until a single "y" or "n" character is input.
    while ( UIinput.upper() != 'Y' and UIinput.upper() != 'N' and arcpy.Exists(ProjectPath + "Final.gdb\\BaseElevation" ) ):
        UIinput = input("\nMosaic Raster Dirctory into new raster? [(Y)es | (N)o] ")

    #Process Raster creating and Mosaic combination if a "Y" character was input.
    if UIinput.upper() == 'Y' or not arcpy.Exists(ProjectPath + "Final.gdb\\BaseElevation" ):
        
        #If the BaseElevation raster exists in the Final Project, delete it.
        if arcpy.Exists(ProjectPath + "Final.gdb\\BaseElevation"): 
            arcpy.Delete_management(ProjectPath + "Final.gdb\\BaseElevation")

        #Create new, empty raster named "BaseElevation" in the Final Project geodatabase.  
        arcpy.CreateRasterDataset_management("Final.gdb","BaseElevation","0.000093","32_BIT_FLOAT",spatialReference,"1","","PYRAMIDS -1 NEAREST JPEG")

        #Read the directory containing the USGS elevation Geotiff 1 degree by 1 degree raster files
        with os.scandir(ProjectPath + "Data\\USGS_1_arc-second\\") as files:
            #Set a holder to the current clock tic to track processing time of loading the rasters and counter of loaded files.
            loadStart = time.perf_counter()
            counterX = 0

            #Loop over each file in the files list.
            for file in files:

                #If file name ends with the .tif extension, process it.
                if file.name.endswith(".tif"):

                    #Create Raster dataset out of current raster file
                    currentRaster = arcpy.sa.Raster(file.path)

                    #determine if current raster dataset intersects with the water basin geometry area of interest.
                    overLap = currentRaster.extent.overlaps(row[0]) or currentRaster.extent.within(row[0])

                    #print current raster file being processed and if it intersects the geometry
                    print(currentRaster.path + '\\' + currentRaster.name + " : " + str(overLap), end=' ')
                    
                    if overLap:
                        #if the raster does intersect the geometry

                        #set local time var to count time taken to load raster in to the BaseElevation raster
                        tic = time.perf_counter()

                        #incriment counter var
                        counterX += 1

                        #Mosaic the current raster into the BaseElevation raster
                        arcpy.Mosaic_management(currentRaster,"Final.gdb\\BaseElevation")

                        #print how long it took to load raster
                        time_format(time.perf_counter() - tic)
                        #print(f"({time.perf_counter() - tic:0.1f} seconds)")
                        
                    else:
                        #if the raster doe not intersect the geometry
                        
                        #print new line character for raster files that were not loaded.
                        print("")

            #Recalculate the statistics to the completed BaseElevation raster
            print("Recalculating Statistics of the BaseElevation Raster")
            arcpy.CalculateStatistics_management("Final.gdb\\BaseElevation", area_of_interest=WBDHU2)

            #Apply Pyramids to the created Base Elevation raster
            arcpy.BatchBuildPyramids_management(ProjectPath + "Final.gdb\\BaseElevation")

            #Print how many files were incorperated into the BaseElevation raster and how long it took to create.
            print(f"{counterX:>3} files loaded in", end=" ")
            time_format(time.perf_counter() - loadStart)

    #Load BaseElevation Dataset
    BaseElevRas = arcpy.sa.Raster("Final.gdb\\BaseElevation")
    
    print("Base Elevation Raster")
    print(BaseElevRas.extent)
    print("")

    print("\n*****************************************************\nCreate Area of Interest Mask from the WBD HU2 Polygon")
    #Ask for User Input on wether to run the create raster out of the Water Basin HU 2 polygon  
    UIinput = ''
    tic = time.perf_counter()
    
    #Keep promting the user until a single "y" or "n" character is input.
    while ( UIinput.upper() != 'Y' and UIinput.upper() != 'N' and arcpy.Exists(ProjectPath + "Final.gdb\\AoI_Mask" ) ):
        UIinput = input("\nCreate Raster from the Water Basin HU2 Polygon? [(Y)es | (N)o] ")

    #Process Raster creating and Mosaic combination if a "Y" character was input.
    if UIinput.upper() == 'Y' or not arcpy.Exists(ProjectPath + "Final.gdb\\AoI_Mask" ):
        
        #If the AoI_Mask raster exists in the Final Project, delete it.
        if arcpy.Exists(ProjectPath + "Final.gdb\\AoI_Mask"): 
            arcpy.Delete_management(ProjectPath + "Final.gdb\\AoI_Mask")

        #Create Raster from the Water Basin HU2 Feature Class  
        arcpy.PolygonToRaster_conversion(ProjectPath + "Data\\WBD_18_HU2_GDB.gdb\\WBD\\WBDHU2","Value","Final.gdb\\AoI_Mask",cellsize=BaseElevRas)

    #Load AoI_Mask Raster Dataset.
    MaskRas = arcpy.sa.Raster("Final.gdb\\AoI_Mask")
    
    print("Area of Interest Raster")
    print(MaskRas.extent)
    time_format(time.perf_counter() - tic)
    print("")

    print("\n***************************************\nCreate Death Valley Mask from HU 8 Polygon")
    #Ask for User Input on wether to run the create raster out of the Death Valley HU 8 polygon  
    UIinput = ''
    tic = time.perf_counter()
    
    #Keep promting the user until a single "y" or "n" character is input.
    while ( UIinput.upper() != 'Y' and UIinput.upper() != 'N' and arcpy.Exists(ProjectPath + "Final.gdb\\DeathValley_Mask" ) ):
        UIinput = input("\nCreate Raster from the Death Valley HU 8 Polygon? [(Y)es | (N)o] ")

    #Process Raster creating and Mosaic combination if a "Y" character was input.
    if UIinput.upper() == 'Y' or not arcpy.Exists(ProjectPath + "Final.gdb\\DeathValley_Mask" ):

        #If Death Valley FC exists, Delete it.
        if arcpy.Exists(ProjectPath + "Final.gdb\\DeathValleyFC"):
            arcpy.Delete_management(ProjectPath + "Final.gdb\\DeathValleyFC")

        #Query string for the death valley polygon in the WBD HU8 Feature set.
        query = """"Name" = 'Death Valley-Lower Amargosa'"""

        #Create new feature set of the Death Valley Polygon
        arcpy.MakeFeatureLayer_management(ProjectPath + "Data\\WBD_18_HU2_GDB.gdb\\WBD\\WBDHU8",ProjectPath + "Final.gdb\\DeathValleyFC",query)

        print(arcpy.GetCount_management(ProjectPath + "Final.gdb\\DeathValleyFC"))
    
        #If the AoI_Mask raster exists in the Final Project, delete it.
        if arcpy.Exists(ProjectPath + "Final.gdb\\DeathValley_Mask"): 
            arcpy.Delete_management(ProjectPath + "Final.gdb\\DeathValley_Mask")

        #Create Raster from the Water Basin HU2 Feature Class  
        arcpy.PolygonToRaster_conversion(ProjectPath + "Final.gdb\\DeathValleyFC","Value","Final.gdb\\DeathValley_Mask",cellsize=BaseElevRas)

    #Load AoI_Mask Raster Dataset.
    DeathValleyRas = arcpy.sa.Raster("Final.gdb\\DeathValley_Mask")
    
    print("Death Valley Raster")
    print(DeathValleyRas.extent)
    time_format(time.perf_counter() - tic)
    print("")

    print("\n********************************\nDecrees Elevation by 200 meters.")

    #Ask for User Input on wether to run minus tool on Base Elevation Raster 
    UIinput = ''
    
    #Keep promting the user until a single "y" or "n" character is input.
    while ( UIinput.upper() != 'Y' and UIinput.upper() != 'N' and arcpy.Exists(ProjectPath + "Final.gdb\\SeaRise" ) ):
        UIinput = input("\nRun Minus tool to change elevation -200? [(Y)es | (N)o] ")

    #Process Raster creating and Mosaic combination if a "Y" character was input.
    if UIinput.upper() == 'Y' or not arcpy.Exists(ProjectPath + "Final.gdb\\SeaRise" ):

        #Set start time var
        tic = time.perf_counter()

        #Apply Minus tool to Base Elevation raster to reduce elevation values by 200 meters
        SeaRiseRas = arcpy.sa.Minus(BaseElevRas,200)

        #If the Sea Rise Elevation raster exists in the Final Project, delete it.
        if arcpy.Exists(ProjectPath + "Final.gdb\\SeaRise"): 
            arcpy.Delete_management(ProjectPath + "Final.gdb\\SeaRise")

        #Save Raster to project geodatabase
        SeaRiseRas.save(ProjectPath + "Final.gdb\\SeaRise")

        #Apply Pyramids to the created Sea Rise raster
        arcpy.BatchBuildPyramids_management(ProjectPath + "Final.gdb\\SeaRise")

        print("Sea Rise Raster")
        print(SeaRiseRas.extent)
        time_format(time.perf_counter() - tic)

    print("\n***************************************\nApply Mask to Crop Sea Rise Raster data")

    #Load Sea Rise Dataset.
    SeaRiseRas = arcpy.sa.Raster("Final.gdb\\SeaRise")

    #Set timer variable
    tic = time.perf_counter()
    
    print("Step 1 - Trim Raster to AoI Mask")

    #Apply Con function to set any value outside the AoI Mask raster to coorsponding Sea Rise Raster cell value rounded to an integer.
    FinalRaster = arcpy.sa.Con( MaskRas, arcpy.sa.Int( SeaRiseRas + 0.5 ) )

    print("Step 2 - Set all cells below 1 to no data")
    
    #Apply con function to set values below zero to null.
    FinalRaster = arcpy.sa.Con( FinalRaster > 0, FinalRaster )

    print("Step 3 - Fill in death valey cells below zero with elevation data")

    #Apply Con function to set null values in the Death Valley region to coorsponding Sea Rise Raster values.
    FinalRaster = arcpy.sa.Con( arcpy.sa.IsNull(DeathValleyRas), FinalRaster, arcpy.sa.Int( SeaRiseRas + 0.5 ) )

    #If the Five Kingdoms Elevation raster exists in the Final Project, delete it.
    if arcpy.Exists(ProjectPath + "Final.gdb\\FiveKingdoms_Elevation"): 
        arcpy.Delete_management(ProjectPath + "Final.gdb\\FiveKingdoms_Elevation")

    #Save Final Raster to project geodatabase as a 16 bit signed raster.        
    arcpy.CopyRaster_management(FinalRaster,ProjectPath + "Final.gdb\\FiveKingdoms_Elevation",pixel_type="16_BIT_SIGNED")
    
    #Apply Pyramids to the created Five Kingdoms raster
    arcpy.BatchBuildPyramids_management(ProjectPath + "Final.gdb\\FiveKingdoms_Elevation")

    print("Final Raster")
    print(FinalRaster.extent)
    time_format(time.perf_counter() - tic)

    print("\n\nFinal Project Process completed in", end=" ")
    time_format(time.perf_counter() - processStart)

except:

    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" + str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
