# Final Project
# GEOG 375, American River College, Spring 2020
#
# Final Project - Mapping USGS Significant Earthquakes
# Created by:  Aleta March
# E-mail:      amarch2u@hotmail.com
# Created on:  04-03-2020
# Updated on:  05-17-2020

'''
PURPOSE OF SCRIPT
 The purpose of this project is to update a cumulative database of significant global earthquake events using data
 retrieved from the USGS Earthquake Hazards Program real-time GeoJSON Summary Format Significant Earthquakes feeds
 and produce a series of maps that includes:
     1) a global map of the location of earthquake events that occurred in the time frame of the real-time feed
        and the time frame of the cumulative database
     2) local maps of the location of individual earthquake events that occurred during the time frame
        (day, week, month) covered by the GeoJSON feed selected by the user.

DATA SOURCES
 1-day  url https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_day.geojson
 7-day  url https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_week.geojson
 30-day url https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_month.geojson

PROGRAM DEFICIENCIES
 Problem 1 Script runtime may lengthen apprciably as number of events in cumulative database increases. 
 Problem 2 Intermittently a point feature does show up on exported PDF map, such as USGS_ID ci39400304
 Problem 3 All the maps have the same spatial reference: GCS WGS 1984, Projection WGS 1984 Mercator Auxiliary Sphere
           Changing the spatial reference of map frames will not be possible with arcpy.mp until ArcGIS Pro 2.6
            
NOTES TO USER
 Updates to cumulative base table with real-time data will only occur if if real-time feed accessed.
   Updates must be made every 30 days or less in order to ensure that the events are not missed.

 In case of an unstable or absent Internet connection, the script will not run, resulting in a message,
   Runtime Error: "Not Signed into Portal". If this occurs, check the Internet connection and re-initiate the script. 

*********************************************************************************************************************
'''
print ('Final Project - Mapping USGS Significant Earthquakes\n')
print ('Create and Update Global Earthquake Geodatabase\n')

import json
import urllib.request, urllib.parse, urllib.error
import arcpy, sys, os, traceback, datetime


try:

    # ===================================SET ENVIRONMENT / OUTPUT PATHS / VARIABLES==================================
    # Set workspace
    arcpy.env.workspace = 'C:\\temp\\Final_Project\\Data\\eq_base.gdb\\'

    # Assign variables

    # Variable for the cumulative base table name
    base_table = 'eq_base_table'

    # Variable for the alternate cumulative base table name (use for testing initial base table creation)
    base_table_alt = 'eq_base_table_alt'

    # Variable for the base table view name
    base_tv = 'eq_base_tv'

    # Variable for the base feature class name
    base_fc = 'eq_base_fc'

    # Variable set to the output path of the file geodatabase
    outpath = 'C:\\temp\\Final_Project\\MyData\\eq_output.gdb'

    # Variable for the output table name
    out_table = 'eq_out_table'

    # Variable that combines the output path and the output table
    eq_out_tbl_path = os.path.join(outpath, out_table)

    # Variable for the output table view name
    out_tv = 'eq_out_tv'

    # Variable for the feature class
    out_fc = 'eq_out_fc'

    # Variable that combines the out path and output fc
    eq_out_fc_path = os.path.join(outpath, out_fc)

    # Variables for segments of the url name used to construct the url 
    url_prefix = 'https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_'
    url_suffix = '.geojson'


    # Initialize variables

    # Count of EQ events in USGS real-time feed
    eq_count = 0
    # List of tuples of row values of events not found in permanent db search
    new_event_rows = []
    # List of tuples of row values of events updated in permanent db search
    updated_event_rows = []
    # Search result indicator
    found = True
    # Script author's name
    author = 'Aleta March'


    # Variables for dates of inception and of most recent update of cumulative database
    db_start_date = '04-16-2020'
    current_date = datetime.datetime.now().strftime('%m-%d-%Y %H:%M:%S')
    print('Permanent database updated on:', current_date)

    
    # =============================================CREATE OUTPUT TABLE===========================================
    # Create output table in file geodatabase and define fields

    # Check to see if output table exists; delete if so
    if arcpy.Exists(eq_out_tbl_path):
        arcpy.Delete_management(eq_out_tbl_path)

    # Create output table
    arcpy.CreateTable_management(outpath, out_table)

    # Add fields to the table

    # USGS GeoJSON id value
    arcpy.AddField_management(eq_out_tbl_path, 'USGS_ID','TEXT', '', '', 15)
    # Date(string)derived from USGS GeoJSON time value
    arcpy.AddField_management(eq_out_tbl_path, 'Date','TEXT', '', '', 10)
    # USGS Event time(string)derived from USGS GeoJSON time value
    arcpy.AddField_management(eq_out_tbl_path, 'Event_Time','TEXT', '', '', 15)
    # USGS GeoJSON mag value
    arcpy.AddField_management(eq_out_tbl_path, 'Magnitude','FLOAT', '4', '2', '')
    # USGS GeoJSON place value
    arcpy.AddField_management(eq_out_tbl_path, 'Place','TEXT', '', '', 50)
    # USGS GeoJSON coordinates[1] value
    arcpy.AddField_management(eq_out_tbl_path, 'Lon','FLOAT', 12, 9, '')
    # USGS GeoJSON coordinates[2] value
    arcpy.AddField_management(eq_out_tbl_path, 'Lat','FLOAT', 11, 9, '')
    # USGS GeoJSON coordinates[3] value
    arcpy.AddField_management(eq_out_tbl_path, 'Depth','FLOAT', 5, 2, '')
    # USGS GeoJSON feed update time value(milliseconds since 1970.01.01 00:00:00)
    arcpy.AddField_management(eq_out_tbl_path, 'Update_Time','FLOAT', 14, 0, '')
    print('Created table and added fields')

    # Create a list of table fields to access with cursors

    field_list = ['USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']
    print('Field List:', field_list, '\n')


    # ==============COPY OUTPUT TABLE TO CUMULATIVE BASE TABLE IF BASE TABLE DOES NOT EXIST==========================

    # If base table does not exist, create it from empty output table
    if arcpy.Exists(base_table) == False:
        arcpy.Copy_management(eq_out_tbl_path, base_table)
        print('Initial cumulative base table established')

    # ==============================RETRIEVE GEOJSON DATA FROM USGS WEBSITE==========================================

    # User enters the USGS desired USGS earthquake feed by timeframe(day, week, or month)
    time_select = input('Enter timeframe of earthquake update(day, week or month): ')

    # If user input incorrect, 
    if time_select != 'day' and time_select != 'week' and time_select != 'month':
        print('USER INPUT ERROR: timeframe options are "day", "week", or "month"')
        
        print('Events recorded for past week will be displayed as default\n')
        time_select = 'week'
       
    url = url_prefix + time_select + url_suffix
    print('Retrieving ', url)


    # Open connection "url handle" to URL
    urlhand = urllib.request.urlopen(url)
    # Read data as bytes and convert to JSON string
    data = urlhand.read().decode()
        
    print('Retrieved ', len(data), 'characters')

    # Convert (deserialize) JSON string to Python dictionary
    jsdata = json.loads(data)

    # Convert to JSON string, then "print it pretty" to examine data structure
    print(json.dumps(jsdata, indent=4))


    # =========================EXTRACT VALUES FROM GEOJSON DATA AND TRANSFER TO OUTPUT TABLE====================

    # Create the insert cursor for the output table
    with arcpy.da.InsertCursor(eq_out_tbl_path, field_list) as irows:

        # Access data from JSON data in Python dictionary format and assign to variables
        for entry in jsdata['features']:

            print('\n')
            eq_id = entry['id']
            print('USGS ID:', eq_id)
            print(type(eq_id))

            # Event origin time (UTC) expressed in milliseconds from midnight 01-01-1970
            time = entry['properties']['time']
            print('UNIX Epoch Time:', time)
            print(type(time))
            epochtime = datetime.datetime(1970, 1, 1) + datetime.timedelta(milliseconds = time)
            print('Date and Time (UTC):', epochtime)
            print(type(epochtime))

            # Convert UNIX epoch time to string
            str_epoch = str(epochtime)
            #print('Epoch time as a string:', str_epoch)

            # Strip date from epoch time string
            date = str_epoch[:10]
            print('Date (UTC):', date)
            print(type(date))

            # Strip event origin time from epoch time string
            pos = str_epoch.find('')
            event_time = str_epoch[pos + 11:]
            print('Event Origin Time (UTC):', event_time)
            print(type(event_time))


            magnitude = entry['properties']['mag']
            print('Magnitude:', magnitude)
            print(type(magnitude))
            place = entry['properties']['place']
            print('Place:', place)
            print(type(place))
            lon = entry['geometry']['coordinates'][0]
            print('Longitude:', lon)
            print(type(lon))
            lat = entry['geometry']['coordinates'][1]
            print('Latitude:', lat)
            print(type(lat))
            depth = float(entry['geometry']['coordinates'][2])
            print('Depth:', depth)
            print(type(depth))
            

            # Convert UNIX epoch millisecond feed update time to float. Integer too large for table field value.
            updated = float(entry['properties']['updated'])
            print('Updated:', updated)

            # Populate row of out_table with data extracted from GeoJSON feed
            irows.insertRow((eq_id, date, event_time, magnitude, place, lon, lat, depth, updated))

            # Increment count of earthquakes
            eq_count = eq_count + 1
    del irows

    #==================================CREATE FEATURE CLASS FROM OUTPUT TABLE===================================
    # Check to see if the Table View already exists, if so, delete it
    if arcpy.Exists(out_tv):
        arcpy.Delete_management(out_tv)

    # Make Table View from the eq_out_table
    arcpy.MakeTableView_management(eq_out_tbl_path, out_tv)

    # Check to see if output feature class of GeoJSON data exists; if so, delete it
    if arcpy.Exists(eq_out_fc_path):
        arcpy.Delete_management(eq_out_fc_path)

    # Create new point feature class based on -x, -y coordinates from table)
    arcpy.management.XYTableToPoint(out_tv, eq_out_fc_path, 'Lon', 'Lat')

    #Print confirmation that new feature class was created
    print('\n' + 'Created new feature class from XY', out_tv,  'to', eq_out_fc_path)

    # Print feature class field list
    print('Feature class field list:', [fld.name for fld in arcpy.ListFields(eq_out_fc_path)])


    print('\n')
    print('Earthquake Count 1-' + time_select + ':', eq_count)
    print('\n')

    # =============================UPDATE CUMULATIVE EARTHQUAKE DATABASE=================================
    # In addition to updating base table with changed attribute values, this code creates:
    #    1) a list of tuples of new event row values
    #    2) a list of tuples of row values that contain updated field values


    # Check to see if permanent base table contains any rows to update: obtain row count 

    # Check to see if the Table View already exists, if so, delete it
    if arcpy.Exists(base_tv):
        arcpy.Delete(base_tv)
    # Make Table View from the base_table (required for subsequent use of GetCount
    arcpy.MakeTableView_management(base_table, base_tv)

    # Obtain row count in the permanent base table and convert it from a UNIX string to an integer
    base_row_count = int(arcpy.GetCount_management(base_tv).getOutput(0))
    print ('Number of entries in permanent base table: ', base_row_count)


    # Compare database created from USGS real-time feed with the permanent cumulative database.
    # Update outdated field values and identify events to be added to the permanent database.

        
    with arcpy.da.SearchCursor(eq_out_tbl_path, field_list) as srows:
        with arcpy.da.UpdateCursor(base_table, field_list) as uprows:               
            search_loop_counter = 0 #Shows current cycle of search loop through real-time feed table
                  
            for srow in srows:      # Assign search row field values used for comparison  
                rt_feed_id = srow[0]       # USGS_ID field value
                rt_feed_updated = srow[8]  # Update_Time field value
                print('Search loop cycle', search_loop_counter)
                
                if base_row_count > 0: # If base table filled, compare entries with real-time feed entries
                    for uprow in uprows:
                        found = False
                        # Set field values to be compared    
                        base_id = uprow[0]
                        base_updated = uprow[8]
                    
                        if base_id == rt_feed_id: # If a match is found
                            found = True
                                                
                            # Check if more recent update available; update base table if so
                            if base_updated < rt_feed_updated:
                                uprows.updateRow(srow)
                                print(base_id, 'base table updated')

                                # Add event row values to updated_event_rows list
                                updated_event_rows.append(srow)
                                print('USGS_ID', rt_feed_id, 'row values added to updated_event_rows list')

                        # If match found, end search
                        if found == True:
                            uprows.reset() # Cursor reset to first row for next loop
                            print('Break out of loop')  # Avoid unnecessary further search
                            break 
                        
                if found == False or base_row_count == 0: # If match not found in permanent db or db empty
                    print('Event not found in permanent database.')
                    # Add to end of list
                    new_event_rows.append(srow)
                    print('USGS_ID', rt_feed_id, 'event row values added to new_event_rows list')
                    print(srow)
                    uprows.reset()
                search_loop_counter = search_loop_counter + 1
                

        del uprows
    del srows

    print('\n')

    #=================================ADD NEW EQ EVENTS TO CUMULATIVE BASE TABLE================================
    # Add entries in new_event_rows to permanent database table, beginning with the last entry in list.
    # This preserves table order: ascending by date and time of event (oldest at the top)

    total_new_events = len(new_event_rows)
    print('Number of rows to be added to base table:', total_new_events)

    if total_new_events > 0: # If there are new events to add to permanent database
        with arcpy.da.InsertCursor(base_table, field_list) as irows:
            index = total_new_events - 1
            while index >= 0:
                # Populate new row of base_table with element (tuple) of new_event_rows
                irows.insertRow(new_event_rows[index])
                
                print('Added new row to base table')
                index = index - 1
        del irows
    print('\n')


    # =====================CREATE CUMULATIVE BASE FEATURE CLASS FROM BASE TABLE IF FC DOES NOT EXIST========================

    # If base feature class does not exist, create it
    if arcpy.Exists(base_fc) == False:
        
        # Check to see if the Table View already exists, if so, delete it
        if arcpy.Exists(base_tv):
            arcpy.Delete_management(base_tv)

        # Make Table View from the eq_out_table
        arcpy.MakeTableView_management(base_table, base_tv)

        # Check to see if output feature class of GeoJSON data exists; if so, delete it
        if arcpy.Exists(base_fc):
            arcpy.Delete_management(base_fc)

        # Create new point feature class based on -x, -y coordinates from table)
        arcpy.management.XYTableToPoint(base_tv, base_fc, 'Lon', 'Lat')

        #Print confirmation that new feature class was created
        print('\n' + 'Created new feature class from XY', base_tv, 'to', base_fc)
        

        # Print feature class field list
        print('Feature class field list:', [fld.name for fld in arcpy.ListFields(base_fc)])

        print('\n')


    # ==========================UPDATE CUMULATIVE BASE FEATURE CLASS ROWS WITH NEW FIELD VALUES =============================
    # Update the base feature class with the row values stored in updated_event_rows list
    # Feature class fields: ['Shape', 'USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']

    total_updates = len(updated_event_rows) 
    if total_updates > 0: # If there are rows to update in base(cumulative) feature class

        # Create update cursor for base feature class; replace 'Shape' field with 'SHAPE@XY' to add 'Point' geometry
        
        with arcpy.da.UpdateCursor(base_fc, ['SHAPE@XY', 'USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']) as uprows:
            index = 0  #Counter for rows to be updated

            for uprow in uprows:
                        
                for index in range(0,total_updates):
                    # If USGS_ID in base_fc matches the ID in the tuple of the updated_event_rows list, then update row
                    if uprow[1] == updated_event_rows[index][0]:
                        # Create new tuple with xy point value as first element (needed to add Point geometry to 'Shape' field)
                        xy = (updated_event_rows[index][5], updated_event_rows[index][6])  # tuple represents (Lon, Lat) values
                        
                        # Create new update row tuple by appending the row elements of updated_event_rows to xy geometry tuple 
                        new_tup = (xy,)
                           
                        for i in range(0,9):
                            new_tup = new_tup + (updated_event_rows[index][i],)
                                           
                        # Update row of base_feature class with element (tuple) of new_tup
                        uprows.updateRow(new_tup)
                        print('Updated row in base feature class', new_tup, '\n')
                        
            del uprow
        del uprows

    #==================================ADD NEW EQ EVENTS TO CUMULATIVE BASE FEATURE CLASS====================================
    # Update the base feature class with the row values stored in new_event_rows list
    # Feature class fields: ['Shape', 'USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']

    if total_new_events > 0: # If there are new events to add to base(cumulative) feature class
        
        # Create insert cursor for base feature class; replace 'Shape' field with 'SHAPE@XY' to add 'Point' geometry
        with arcpy.da.InsertCursor(base_fc, ['SHAPE@XY', 'USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']) as irows:
            
            index = total_new_events - 1  #Counter for rows to be added
            while index >= 0:
            
                # Create new tuple with xy point value as first element (needed to add Point geometry to 'Shape' field)
                xy = (new_event_rows[index][5], new_event_rows[index][6])  # tuple represents (Lon, Lat) values

                # Create new row tuple by appending the row elements of new_event_rows to xy geometry tuple 
                new_tup = (xy,)
                for i in range(0,9):
                    new_tup = new_tup + (new_event_rows[index][i],)
                
                # Populate new row of base_feature class with element (tuple) of new_tup
                irows.insertRow(new_tup)

                # Decrement index
                index = index - 1
                print('Added new row to base feature class', new_tup, '\n')
                
        del irows

    print('\n')

    #============================================PART II: CREATE AND EXPORT MAP SERIES=======================================

    '''
    **************************************************OVERVIEW*******************************************************

    Output maps are variations of two types: global and local
       1) one global map at full map extent showing two layers (cumulative overlaid by 1, 7, or 30 day real-time feed)
            with graduated point symbols based on EQ event magnitude
    
       2) a group of local maps equivalent to the number of real-time events showing the real-time feed layer
            zoomed-in to each point feature(or some specified distance around the individual eq event location)
    *******************************************************************************************************************

    '''
    print ('Create Map Series from Earthquake Geodatabase\n')

    '''
    # =================================INCLUDED TO ALLOW MAP SERIES SCRIPT TO RUN INDEPENDENTLY=========================
    import json
    import urllib.request, urllib.parse, urllib.error
    import arcpy, sys, os, traceback, datetime

    # Variables for dates of inception and of most recent update of cumulative database
    db_start_date = '04-09-2020'
    current_date = datetime.datetime.now().strftime('%m-%d-%Y %H:%M:%S')
    # Script author's name
    author = 'Aleta March'

    # User input for time frame of real-time feed
    time_select = 'week'

    # ===================================================================================================================
    '''

    # ===============================SET PATHS FOR PROJECT ACCESS AND MAP EXPORT=========================================
    # Output path for PDFs
    map_out_path = 'C:\\temp\\'

    # Variable set to the location of the ArcGIS project folder
    pfolder_path = 'C:\\Users\\Mercedes\\Documents\\ArcGIS\\Projects\\'

    # Variable set to the name of the current project location
    cur_proj = 'eq_geojson_2db\\eq_geojson_2db_mp2.aprx'

    # Variable set to full project path
    full_ppath = os.path.join(pfolder_path, cur_proj)

    print('Full path to map:', full_ppath)

    # Variable used to access project object
    aprx = arcpy.mp.ArcGISProject(full_ppath)

    # Variables set to map object(s) in list of maps, layouts, layers and map elements; [0] is first in list
    maplist = aprx.listMaps('EQ Event Map')[0]        # Map tab name is 'EQ Event Map'; unique map name
    layout = aprx.listLayouts('Landscape Layout')[0]  # Layout tab name is 'Landscape Layout'
    mapframe = layout.listElements('MAPFRAME_ELEMENT', 'Map Frame')[0]       # Map frame name for global map
    # Map frame specific to spatial reference not in use in script; initial attempt to provide >1 spatial reference  
    mapframe_sr = layout.listElements('MAPFRAME_ELEMENT', 'Map Frame US')[0]
    legend = layout.listElements('LEGEND_ELEMENT', 'Legend')[0]
    txt_elems = layout.listElements('TEXT_ELEMENT')   # References list of text element objects
    lyr_list = maplist.listLayers()                   # List of layer objects in 'EQ Event Map'


    # Print map names and layer names (feedback) 
    for proj_map in aprx.listMaps():
        print("Map: " + proj_map.name)
        for lyr in proj_map.listLayers():
            print("  " + lyr.name)
              
    # ========================================PREPARE MAP LAYOUTS AND EXPORT MAPS========================================
    # Retrieve a list of layers and assign variable to each layer
    for TOCLayer in lyr_list:
        
        # Access base topographic map layer and assign variable
        if TOCLayer.name == 'World Topographic Map':
            base_topo_lyr = TOCLayer
                        
        # Access cumulative earthquake map layer and assign variable
        if TOCLayer.name == 'eq_base_fc':
            cumlatv_EQ_lyr = TOCLayer
                
        # Access real-time earthquake map layer and assign variable
        if TOCLayer.name == 'eq_out_fc':
            rt_lyr = TOCLayer
            
        
    # Establish map element values used by all maps in series or assign variables

    # Retrieve list of text elements
    # Set text element values that remain constant for all maps in series
    for txt_elem in txt_elems:
                        
        if txt_elem.name == 'Print Date':
            txt_elem.text = 'Printed: ' + str(current_date)

        if txt_elem.name == 'Author':
            txt_elem.text = str(author)

        if txt_elem.name == 'Description':
            txt_elem.text = 'EQ Magnitude (Mw)'

        if txt_elem.name == 'Info':
            txt_elem.text = 'Moment Magnitude (Mw): measure of total energy released by earthquake at its source; logarithmic scale of 1-10'

        if txt_elem.name == 'Credits':
            txt_elem.text = 'Data Source: "U.S.Geological Survey Significant Earthquakes",[real-time data feed]. Retrieved from https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/  significant_month.geojson.'

        if txt_elem.name == 'Spatial Ref':
            txt_elem.text = 'GCS: WGS 1984' + '\n' + 'Projection: WGS 1984 Mercator Auxiliary Sphere' 


        # Assign variables to text elements whose value or position differ between maps in series 
        if txt_elem.name == 'Title':
            map_title = txt_elem

        if txt_elem.name == 'Subtitle':
            map_subtitle = txt_elem
                            
        if txt_elem.name == 'Time Frame':
            time_frame = txt_elem

        if txt_elem.name == 'RT Info':
            rt_info = txt_elem

        if txt_elem.name == 'Info':
            info = txt_elem

        
    # Assign value to time frame
    if time_select == 'day':
        time_frame.text = '1 Day'
    elif time_select == 'week':
        time_frame.text = '7 Days'
    elif time_select == 'month':
        time_frame.text = '30 Days'
    else: time_frame.text = '7 Days'

    # Set time frame
    time_frame.text = 'Real-time Feed:' + '\n' + 'Past ' + str(time_frame.text) + '\n \n' + 'Cumulative Period:' + '\n' + str(db_start_date) + ' to Present'


    # Amend Layout for Global Map of Cumulative and Real-time Events

    # Set map elements specific to global map
    # Set map title and subtitle
    map_title.text = 'USGS Significant Earthquakes'
    print('Map title on layout is now:', map_title.text)
    print('\n')
    map_subtitle.text = db_start_date + ' to Present'

    # Set RT Info text element invisible
    rt_info.visible = False

    # Set position of RT Info textbox (move up out of way as not needed) 
    textbox_pos_y = 2.2552
    info.elementPositionY = textbox_pos_y

    # Set vertical position of time_frame text box (move up to fill Timeframe textbox) 
    textbox_pos_y = 2.8739
    time_frame.elementPositionY = textbox_pos_y
       
    # Make spatial reference specific mapframe(s) invisible
    mapframe_sr.visible = False
        
    # Set extent to that of global base map    
    layer_extent = mapframe.getLayerExtent(base_topo_lyr, True, True)
    print('Layer extent:'  + str(layer_extent))
    mapframe.camera.setExtent(layer_extent)

                  
    # Export map
    if arcpy.Exists(map_out_path + 'Global_Test_Map.pdf'):
        arcpy.Delete_management(map_out_path + 'Global_Test_Map.pdf')     
    layout.exportToPDF(map_out_path + 'Global_Test_Map.pdf')

    print('\n')


    # Amend Layout for Map of Individual Real-time EQ Event Locations
        
    if eq_count > 0: # If there are real-time events to map
           
        # Retrieve attribute values from real-time layer of 'EQ Event Map'              
        with arcpy.da.SearchCursor(rt_lyr, ['Shape', 'USGS_ID', 'Date', 'Event_Time', 'Magnitude', 'Place', 'Lon', 'Lat', 'Depth', 'Update_Time']) as rt_rows:
            for rt_row in rt_rows:
                            
                # Set variables for current real-time EQ event attributes
                rt_USGS_ID = rt_row[1]
                rt_date = rt_row[2]
                rt_event_time = rt_row[3]
                rt_magnitude = round(rt_row[4], 1)
                rt_place = rt_row[5]
                
                # Format Layout for individual RT event       
                # Convert rt_date string Y-m-d to month-day-year format
                rt_conven_date = rt_date[5:] + '-' + rt_date[:4]
                          
                # Change value of map title and decrease font size for lengthy titles
                map_title.text = rt_place
                print('Map title on layout is now:', map_title.text)
                print('\n')
                            
                if len(map_title.text) > 24:
                    map_title.textSize = 30

                # Control text element visibility (show RT date and magnitude, remove subtitle)
                rt_info.visible = True
                map_subtitle.visible = False

                # Set RT Info text 
                rt_info.text = 'Event Date (UTC):' + '\n' + rt_conven_date + '\n' + 'Magnitude:' + '\n' + str(rt_magnitude) + ' Mw'

                # Set position of time_frame textbox (move down to make room for RT Info textbox
                textbox_pos_y = 1.9762
                time_frame.elementPositionY = textbox_pos_y

                # Set position of info textbox
                textbox_pos_y = 1.3575
                info.elementPositionY = textbox_pos_y

                                                                
                # SUBSET REAL-TIME LAYER TO INDIVIDUAL EARTHQUAKE EVENT        
                # Select current row for definition query
                query = """"USGS_ID" = '""" + rt_USGS_ID +"""'"""
                arcpy.SelectLayerByAttribute_management(rt_lyr, "NEW_SELECTION", query)
                rt_lyr.definitionQuery = query
                print(rt_USGS_ID, 'selected to map')
                        
                # Get/Set layerExtent for selected row   
                layer_extent = mapframe.getLayerExtent(rt_lyr, True, True) # Booleans refer to selected extent and symbolized extent 
                print('Layer extent rt_layer:'  + str(layer_extent))
                                
                # Set mapframe camera to layer extent
                mapframe.camera.setExtent(layer_extent)
                # Set mapframe scale (create zoom-in effect)
                mapframe.camera.scale = 2400000.00

                            
                # Modify spatial reference of mapframe
                #   NOT POSSIBLE: according to Jeff Barrette, ESRI arcpy.mp team, this will be available in ArcGIS Pro 2.6
                

                # Export map of one real-time event
                # Clear selected features        
                arcpy.SelectLayerByAttribute_management(rt_lyr, "CLEAR_SELECTION")

                # Delete pdf map with identical name, if exists
                if arcpy.Exists(map_out_path + rt_USGS_ID + '_map.pdf'):
                    arcpy.Delete_management(map_out_path + rt_USGS_ID + '_map.pdf')
                # Export to pdf                
                layout.exportToPDF(map_out_path + rt_USGS_ID + '_map.pdf')
                print('\n')

            del rt_row           
        del rt_rows

    
    print('\n' + 'End')

except:
    # try:/except: code taken directly from ARC GEOG 375 SP20 class materials "Demo 6D - Create and Use Table Joins"
    #    authored by Nathan Jennings, 21 Feb. 2020, Copyright 2011-2020. Author referenced the following ESRI website:
    #    https://pro.arcgis.com/en/pro-app/arcpy/get-started/error-handling-with-python.htm

    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +  str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)

    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
