# GEOG 375 Final Project

# Project Title: COVID-19 Testing Locations in California Map Series

# Created by: Cameron Gee

# Created on: 04.29.2020
# Updated on: 05.17.2020

'''
This script produces a series of maps showing COVID-19 testing locations in the state of California.
A project file is accessed which contains a layout template as the basis for each automated map.
A layer with United States testing locations is accessed and queried to show only the locations
which are currently open in California. A 3-digit zip code layer defines the number of maps
the script will produce. Each map zooms in to a 3-digit zip code area and shows the nearby
testing locations. Background layers such as highways, hydrology, counties, states, and countries
are provided in the project file.
'''

# 1. Import arcpy, sys, traceback, and datetime

import arcpy, sys, traceback, datetime


# 2. Create workspace and define author and current date variables to be used later

arcpy.env.workspace = r"C:\Users\camer\Documents\_School2\GEOG 375\Final Project\\"
author = "Cameron Gee"
cur_date = datetime.date.today().strftime("%m.%d.%Y")


try:

    # 3. Define paths for the project folder, aprx file, data folder, and export folder to access.
    #    Define a variable for the California 3-digit zip codes layer to be zoomed in on in each map

    projectpath = r"C:\Users\camer\Documents\_School2\GEOG 375\Final Project\Final Project\\"
    aprx = arcpy.mp.ArcGISProject(projectpath + "Final Project.aprx")
    datapath = r"C:\Users\camer\Documents\_School2\GEOG 375\Final Project\Final Project\Data\\"
    exportpath = r"C:\Users\camer\Documents\_School2\GEOG 375\Final Project\PDFs\\"
    zip3_CA = datapath + "zip3_CA.shp"
       
   
    # 4. Get a list of maps, layouts, and elements to access

    maplist = aprx.listMaps("Map")[0]
    layout = aprx.listLayouts("Layout")[0]
    mapframe = layout.listElements("MAPFRAME_ELEMENT", "Map Frame")[0]


    # 5. Get a list of layers to access and modify

    print("Accessing a list of layers in the project file...")
    for TOCLayer in maplist.listLayers():
        print("Layer Name: " + str(TOCLayer.name))


        # 6. Access the Testing Locations layer inside the map's table of contents
        #    and select the locations that are within California and currently open.
        #    Print the results
          
        if TOCLayer.name == "Testing_Locations":
            query_CA = """"State" = 'CA'"""
            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query_CA)
            result = arcpy.GetCount_management(TOCLayer)
            print("Number of testing locations that are in California: " + str(result))

            query_open = """"status" = 'Open'"""
            arcpy.SelectLayerByAttribute_management(TOCLayer, "SUBSET_SELECTION", query_open)
            result = arcpy.GetCount_management(TOCLayer)
            print("Number of testing locations that are in open in California: " + str(result))


            # 7. Perform a definition query on the selected locations and clear the selection
        
            TOCLayer.definitionQuery = query_CA + "AND" + query_open
            print("Created definition query for Testing_Locations layer")
            arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")


            # 8. Turn on labels for the resulting testing locations

            TOCLayer.showLabels = True


        # 9. Access the 3-digit zip codes layer inside the map's table of contents
        #    and select the zip codes that are within California. This will determine
        #    the number of maps to generate. Print the results

        if TOCLayer.name == "zip3":
            query_zip = """"STATE" = 'CA'"""
            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query_zip)
            result = arcpy.GetCount_management(TOCLayer)
            print("Number of 3-digit zip code areas that are in California: " + str(result))
            print("Number of maps to be drawn: " + str(result))


            # 10. Check to see if a shapefile exists for the California 3-digit zip codes.
            #     If a shapefile already exists, delete it. Copy the selected features from above
            #     to a new shapefile

            if arcpy.Exists(zip3_CA):
                arcpy.Delete_management(zip3_CA)
            arcpy.CopyFeatures_management(TOCLayer, zip3_CA)

            
        # 11. Define the TOCLayer as the California 3-digit zip codes layer before operating on each map
            
        if TOCLayer.name == "zip3_CA":
            ZipLayer = TOCLayer
        
            
    # 12. Create a search cursor to identify a list of all California 3-digit zip codes

    field_name = ["ZIP3"]
    with arcpy.da.SearchCursor(ZipLayer, field_name) as ziprows:
        for ziprow in ziprows:
            zipName = ziprow[0]
            query = """"ZIP3" = '""" + zipName + """'"""
            print("Generating map for: " + query)
            ZipLayer.definitionQuery = query


            # 13. For each map, set the extent to each 3-digit zip code in the list

            layer_extent = mapframe.getLayerExtent(ZipLayer)
            mapframe.camera.setExtent(layer_extent)
            mapframe.camera.scale = mapframe.camera.scale * 1.04


            # 14. For each map, edit the following elements:

            tElements = layout.listElements("TEXT_ELEMENT")
            for tElement in tElements:
                if tElement.name == "Map Title":
                    tElement.text = "COVID-19 Testing Locations"
                if tElement.name == "Zip 3":
                    tElement.text = "California 3-Digit Zip Code: " + zipName
                if tElement.name == "Author":
                    tElement.text = "Author: " + author
                if tElement.name == "Print Date":
                    tElement.text = "Print Date: " + str(cur_date)


            # 15. Check to see if a PDF exists for each map. If a PDF already exists, delete it.
            #     Export the layout of each map as a PDF
                    
            if arcpy.Exists(exportpath + zipName + "_map.pdf"):
                arcpy.Delete_management(exportpath + zipName + "_map.pdf")
            layout.exportToPDF(exportpath + zipName + "_map.pdf")
            print("Created " + zipName + "_map.pdf")
    print("Finished generating maps to " + exportpath)


# Add exception code within except block

except:
    # https://pro.arcgis.com/en/pro-app/arcpy/get-started/error-handling-with-python.htm
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" + str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
