# Python CNDDB Script
# Final Project
# Created by: Greg Ewing
# Created on: 04/24/2020

''' 
This script was designed to assist the GIS Analyst in creating a CNDDB (California Natural
Diversity Database) feature class output based on a user defined distance from a spill site.  
The purpose of this script allows the GIS Analyst to provide timely and relevant information 
of what threatened and endangered species are likely to exist in the surrounding spill site area.

This custom script was made from the Buffer and Clip Analysis tools respectively, whose inputs
come from a tool interface.  These inputs from the interface are introduced into the code
by the use of the "GetParameterAsText" function which specifies the parameter based on an index 
number.  Also to add the Buffer and Clip structure was influenced by the 2011 Chapter 10 in-class 
Clip and Buffer sample code provided by Nathan Jennings.

Additionally, messages are provided via the AddMessage function to keep the user informed to
the codes progress. '''

# imports the needed modules
import arcpy, sys, traceback

# All portions of the code, commented out in green below, are no longer needed.  This 
# part of the code will be defined by the user interface inputs from the ArcToolbox script tool.  

'''
# Defines the current workspace.  User needs to create a geodatabase prior to running this script.
# arcpy.env.workspace = "C:\\Tmp\\Python_Project\\Python_Data\\Python_Final_Project.gdb\\Source_Data\\"

# Defines the output path.
# output_path = "C:\\Tmp\\Python_Project\\Python_Data\\Python_Final_Project.gdb\\Script_Results\\"
'''


# Defines the variables to the Buffer tool.

# Defines the input feature Layer. 
''' input_spillsite = "CuyamaRiver_SpillSite" '''
input_spillsite = arcpy.GetParameterAsText (0)

# Assigns the buffer values to 1, 3, 5, 10, or 15.
''' buff_values = 3 '''
buff_values = arcpy.GetParameterAsText (2)

# assigns the buffer units to Miles, Feet, or Meters.
''' buff_unit = "Miles" '''
buff_unit = arcpy.GetParameterAsText (3)

# Defines the Buffer distance input (value and units).
buffer_distance = str(buff_values) + " " + buff_unit

# Defines the unique name extention for the Buffer and Clip results.  
# No longer needed - Defined by user interface. 
''' buff_name = "_" + str(buff_values) + buff_unit '''

# Defines the Output Feature Class by combining the output path and the new 
# naming of input spill site.  No longer needed - Defined by interface.
''' output_spillsite = os.path.join(output_path, input_spillsite + buff_name) '''
output_spillsite = arcpy.GetParameterAsText (1)



# Defines the variables to the Clip tool.

# Defines the input Feature Layer.
''' input_CNDDB = "CNDDB" '''
input_CNDDB = arcpy.GetParameterAsText (4)

# Defines the "clip_feature" used in the clip tool.  This is the output feature
# Class of Buffer Tool - "output_spillsite".  Define by user interface.

# Defines the output Feature Class.  This is the desired result used for mapping purposes.
''' output_CNDDB = os.path.join(output_path, input_CNDDB + buff_name) '''
output_CNDDB = arcpy.GetParameterAsText (5)


try:
    
    # Checks to see if site spill data exists, if so delete it.
    if arcpy.Exists(output_spillsite):
        arcpy.Delete_management(output_spillsite)
     
    # The Multiple Ring Buffer (MRB) tool was originally going to be used as the first tool.
    # Between errors that were being generated on the output that could not be resolved and realizing
    # that a different tool was better suited for the desired goal, the MRB tool was replaced 
    # for the Buffer tool.

    
    # The code below for the Buffer and Clip Analysis tools, was 
    # influenced by the 2011 Chapter 10 in-class Clip and Buffer sample code provided by Nathan Jennings.
        
    # Starting Buffer routine on the spill site data.  
    
    # Adds informative messaging to the Python Script toolbox. 
    print ("Processing spill site data using the Buffer tool.  Please wait...\n")
    arcpy.AddMessage("Processing spill site data using the Buffer tool.  Please wait...\n")
    
    print ("Buffering input " + input_spillsite + " by a distance of " + buffer_distance + ".\n")
    arcpy.AddMessage("Buffering input " + input_spillsite + " by a distance of " + buffer_distance + ".\n")
           
    # Executes Buffer Tool at specified distance around the input feature.  Defined
    # by user interface.     
    arcpy.Buffer_analysis(input_spillsite, output_spillsite, buffer_distance)
    
    # Adds informative messaging to the Python Script toolbox.   
    print ("Output buffer file: " + output_spillsite + "\n")
    arcpy.AddMessage("Output buffer file: " + output_spillsite + "\n") 
    
    print ("Buffer tool processing complete!!\n")
    arcpy.AddMessage("Buffer tool processing complete!!\n")
    
    # Checks to see if CNDDB data exists, if so delete it.
    if arcpy.Exists(output_CNDDB):
        arcpy.Delete_management(output_CNDDB)
        
        
    # Starting Clip routine on the CNDDB data.
    
    # Adds informative messaging to the Python Script toolbox.
    print ("Processing the statewide CNDDB data using the Clip tool.  Please wait...\n")
    arcpy.AddMessage("Processing the statewide CNDDB data using the Clip tool.  Please wait...\n")
    
    print ("Clipping " + input_CNDDB + " using " + output_spillsite + " buffer output\n")
    arcpy.AddMessage("Clipping " + input_CNDDB + " using " + output_spillsite + " buffer output\n")
    
    # Executes Clip Tool using resultant of buffer tool.  Defined by user interface.
    arcpy.Clip_analysis(input_CNDDB, output_spillsite, output_CNDDB)
    
    # Adds informative messaging to the Python Script toolbox.
    print ("Output Clip file: " + output_CNDDB + "\n")
    arcpy.AddMessage("Output Clip file: " + output_CNDDB + "\n")
    
    print ("Clipping tool processing complete!\n")
    arcpy.AddMessage("Clipping tool processing complete!\n")

    print ("Processing of data complete!!!")
    arcpy.AddMessage("Processing of data complete!!!\n")
        
        
except:
    
    # https://pro.arcgis.com/en/pro-app/arcpy/get-started/error-handling-with-python.htm
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" + str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
