#Final Project, Geog 375
#ARC, Spring 2020
#Created by: Austin Demshar
#Created on: 4/17/2020
#Updated on: 5/14/2020

"""
This Python script, designed for ArcGIS Pro, will utilize multiple shapefile datasets to return geodatabase and excel data
(with some intermediate derived datasets) that detail hiking trails within the Folsom and Auburn State Recreation Areas, 
within a distance of a queried addresss. The data will include other relevant details the user may find useful, such as trail 
distances, as well as closest river, water body, and associated distances (in feet, miles, or both). It is not currently 
designed to work within an ArcGIS custom toolbox, but run as a standalone script within a Python IDLE (developed in Spyder 3.3.6).

For customizability within this script, see:
> query_a = adjust starting address
> query_b = adjust park/park system within the state
> SelectLayerByLocation = adjust search radius from address

"""

#Import the necessary modules for the project
import arcpy, os, sys, traceback

#Define the workspaces (filepath and geodatabase)
arcpy.env.workspace = r'C:\Personal\ARC GIS Coursework\Geog 375 - Programming GIS\Final Project\Workspace\Final_Project.gdb'
gdb_path = arcpy.env.workspace
shapefile_path = r'C:\Personal\ARC GIS Coursework\Geog 375 - Programming GIS\Final Project\Workspace'

##Preliminary variables defined

#Original shapefiles for the project
trails_shp = os.path.join(shapefile_path, 'RecreationalRoutes.shp')          
rivers_shp = os.path.join(shapefile_path, 'NHD_MajorRiversAndCreeks.shp')
lakes_shp = os.path.join(shapefile_path, 'NHD_MajorLakesAndReservoirs.shp')
addresses_shp = os.path.join(shapefile_path, 'Address.shp')

#Projection alignment variables
PCS_target = arcpy.SpatialReference('NAD 1983 StatePlane California II FIPS 0402 (US Feet)')
#GCS_WGS_1984 = arcpy.SpatialReference('WGS 1984')
#GCS_NAD_1983 = arcpy.SpatialReference('NAD 1983')
GCS_trans = 'WGS_1984_(ITRF00)_To_NAD_1983' #transformation required for certain datasets
    
#Layer and feature class variables
trails_projected = 'trails_projected'
rivers_projected = 'rivers_projected'
lakes_projected = 'lakes_projected'

trails_lyr_pr = 'trails_projected_layer'
rivers_lyr_pr = 'rivers_projected_layer'
lakes_lyr_pr = 'lakes_projected_layer'
addresses_lyr = 'addresses_layer'
trails_join_lyr = 'trails_join_layer'

trails_dissolved = 'trails_dissolved'
trails_join_a = 'trails_river_join'
trails_join_b = 'trails_river_water_join'

trails_table = 'SRA_Trails_Table'
trails_xls_path = os.path.join(shapefile_path, 'SRA_Trails_Table.xls')
trails_FC = 'trails_FC'

#Start geoprocessing and various ArcGIS tasks
try:
    
    print(">>> Begin script <<<")
    
    print("Starting Folsom/Auburn SRA Trails analysis script...")
    
    print("Starting geoprocessing tasks...")

    #Check workspace for projected shapefiles, delete if so
    if arcpy.Exists(trails_projected):
        arcpy.Delete_management(trails_projected)
    if arcpy.Exists(rivers_projected):
        arcpy.Delete_management(rivers_projected)
    if arcpy.Exists(lakes_projected):
        arcpy.Delete_management(lakes_projected)
        
    #Project datasets to same CS (transformation included if needed; addresses already in target projected CS)
    arcpy.Project_management(trails_shp, trails_projected, PCS_target, GCS_trans)
    arcpy.Project_management(rivers_shp, rivers_projected, PCS_target)
    arcpy.Project_management(lakes_shp, lakes_projected, PCS_target)

    print("All required datasets projected to State Plane, California, Zone 2")
    
    #Check workspace for projected feature layers, delete if so
    if arcpy.Exists(trails_lyr_pr):
        arcpy.Delete_management(trails_lyr_pr)
    if arcpy.Exists(rivers_lyr_pr):
        arcpy.Delete_management(rivers_lyr_pr)
    if arcpy.Exists(lakes_lyr_pr):
        arcpy.Delete_management(lakes_lyr_pr)
    if arcpy.Exists(addresses_lyr):
        arcpy.Delete_management(addresses_lyr)

    #Make feature layers for all projected datasets
    arcpy.MakeFeatureLayer_management(trails_projected, trails_lyr_pr)
    arcpy.MakeFeatureLayer_management(rivers_projected, rivers_lyr_pr)
    arcpy.MakeFeatureLayer_management(lakes_projected, lakes_lyr_pr)
    arcpy.MakeFeatureLayer_management(addresses_shp, addresses_lyr)

    #Define various query variables to be used in the selection geoprocessing tools below
    query_a = """"Address_Nu" = '4661' AND "Street_Nam" = 'PLANTATION'"""
    query_b = """("UNITNAME" = 'Auburn SRA' OR "UNITNAME" = 'Folsom Lake SRA') AND "ROUTECLASS" = 'State Park Trail'"""
    
    #Select by attributes, location, and then dissolve, printing statements after various processes complete
    arcpy.SelectLayerByAttribute_management(addresses_lyr, "NEW_SELECTION", query_a)
    arcpy.SelectLayerByAttribute_management(trails_lyr_pr, "NEW_SELECTION", query_b)
    trails_count = arcpy.GetCount_management(trails_lyr_pr)

    print("Trail count after attribute queries for hiking trails within Folsom/Auburn SRA = " + str(trails_count))
    
    #Spatial selection limited to 30 miles, subsetting selection from above
    arcpy.SelectLayerByLocation_management(trails_lyr_pr, "WITHIN_A_DISTANCE", addresses_lyr, "30 Miles", "SUBSET_SELECTION")
    trails_count = arcpy.GetCount_management(trails_lyr_pr)

    print("Trail count after spatial query considering a 30 mile radius from selected address = " + str(trails_count))

    if arcpy.Exists(trails_dissolved):
        arcpy.Delete_management(trails_dissolved)

    #Dissolve trails selection by route name and sum the length of all segments per trail
    arcpy.Dissolve_management(trails_lyr_pr,trails_dissolved,"ROUTENAME",("SEGLNGTH SUM"))
    dissolve_count = arcpy.GetCount_management(trails_dissolved)
    
    #Add and calculate field for all trail spans (dissolved from multiple trail segments)
    arcpy.AddField_management(trails_dissolved, "trail_span", "TEXT")
    arcpy.management.CalculateField(trails_dissolved, "trail_span", "IIf($feature.SUM_SEGLNGTH / 5280 > 1, \
                                                                         round($feature.SUM_SEGLNGTH / 5280, 2) + ' Miles', \
                                                                         $feature.SUM_SEGLNGTH + ' Feet')", "ARCADE", '', "FLOAT")

    print("Trail count of Folsom/Auburn SRA trails, dissolved to entire trail = " + str(dissolve_count))
    
    if arcpy.Exists(trails_join_a):
        arcpy.Delete_management(trails_join_a)
    if arcpy.Exists(trails_join_b):
        arcpy.Delete_management(trails_join_b)
    
    #Spatial joins to bring together closest river and water body, along with associated distances to trails
    arcpy.SpatialJoin_analysis(trails_dissolved, rivers_lyr_pr, trails_join_a, "JOIN_ONE_TO_ONE","#","#", "CLOSEST", "#", "riv_prox_j")
    arcpy.SpatialJoin_analysis(trails_join_a, lakes_lyr_pr, trails_join_b, "JOIN_ONE_TO_ONE","#","#", "CLOSEST", "#", "wtr_prox_j")
                               
    #Add and calculate fileds for river and water body proximity (rivers in feet, lakes dynamic for feet or miles)
    arcpy.AddField_management(trails_join_b, "riv_prox", "FLOAT")
    arcpy.CalculateField_management(trails_join_b, "riv_prox", "round((!riv_prox_j!),2)")

    arcpy.AddField_management(trails_join_b, "wtr_prox", "TEXT")
    arcpy.CalculateField_management(trails_join_b, "wtr_prox","IIf($feature.wtr_prox_j / 5280 > 1, \
                                                                   round($feature.wtr_prox_j / 5280, 2) + ' Miles', \
                                                                   Round($feature.wtr_prox_j,0) + ' Feet')", "ARCADE", '', "FLOAT")
                
    print("Spatial joins completed to incorporate river and water body information")

    if arcpy.Exists(trails_join_lyr):
        arcpy.Delete_management(trails_join_lyr)
    
    #Feature layer created from final trails selection and joined rivers, lakes information
    arcpy.MakeFeatureLayer_management(trails_join_b, trails_join_lyr)
    
    #Near analysis to associate distance from queried address to trails information
    arcpy.Near_analysis(trails_join_lyr, addresses_lyr)

    arcpy.AddField_management(trails_join_lyr, "near_miles", "FLOAT","","")
    arcpy.CalculateField_management(trails_join_lyr, "near_miles","round(!NEAR_DIST! / (5280),2)")

    print("Near analysis completed and joined to trails/river/waterbody data")
    
    #The following section details final outputs: create table, feature class, and excel file (selected fields only)
    if arcpy.Exists(trails_table):
        arcpy.Delete_management(trails_table)  
    if arcpy.Exists(trails_xls_path):
        arcpy.Delete_management(trails_xls_path)  
    if arcpy.Exists(trails_FC):
        arcpy.Delete_management(trails_FC)
        
    #Create table and associated fields to be used
    arcpy.CreateTable_management(gdb_path, 'SRA_Trails_Table')
    arcpy.AddField_management(trails_table, "ROUTENAME", "TEXT","","","","Trail name")
    arcpy.AddField_management(trails_table, "near_miles", "FLOAT","","","","Distance to trail (miles)")
    arcpy.AddField_management(trails_table, "trail_span", "TEXT","","","","Trail length (feet or miles)")
    arcpy.AddField_management(trails_table, "GNIS_Name", "TEXT","","","","Nearest river")
    arcpy.AddField_management(trails_table, "riv_prox", "FLOAT","","","","Nearest river (feet)")
    arcpy.AddField_management(trails_table, "GNIS_Name_1", "TEXT","","","","Nearest lake")
    arcpy.AddField_management(trails_table, "wtr_prox", "TEXT","","","","Nearest lake (feet or miles)")
   
    #Table field list variable for output table and feature class, ordered
    table_f_list = ["ROUTENAME", "near_miles", "trail_span", "GNIS_Name", "riv_prox", "GNIS_Name_1", "wtr_prox"]
    
    print("Table created")

    #Create search cursor and insert cursor to populate new table with fields from geoprocessed layer
    with arcpy.da.SearchCursor(trails_join_lyr, table_f_list) as srows:
        with arcpy.da.InsertCursor(trails_table, table_f_list) as irows:
            for srow in srows:
                irows.insertRow(srow)
            
    #Create field mappings from table, and create feature class with mapping
    field_mappings = arcpy.FieldMappings()  
    field_mappings.addTable(trails_table)
    arcpy.FeatureClassToFeatureClass_conversion(trails_join_lyr, gdb_path, "trails_FC","",field_mappings)
    
    print("Feature class created")
    
    #Create excel output for ease of access (accessing GIS software isn't necessary to view the excel table)
    ##Python error: note in the shell that this process returns a "soft error", though the table is properly exported
    ##Error is: AttributeError: 'ToolValidator' object has no attribute 'isLicensed'
    ##The issue is currently under review with ESRI, see: https://support.esri.com/en/bugs/nimbus/QlVHLTAwMDEyNDI4Ng== 
    arcpy.TableToExcel_conversion(trails_table, trails_xls_path, "ALIAS") 

    print("Table exported to excel (SRA_Trails_Table.xls), to the workspace")      
   
    print("<<< End script >>>")

except:

    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n " + str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
