# Create a Python list of unique values
# for the Amenity field in the Park Amenities

# Created by:  Nathan Jennings
# Created on: 04.16.2021
# Updated on: 04.16.2021
# UrbandaleSpatial.com

# AGOL City of Sacramento Park Amenities layer
# (i.e. City of Sacramento publishes the Park Amenities
# to ArcGIS Online (shared to Everyone - i.e. full public access)

# NOTE:  Normally, working with feature services
# one can use the ArcGIS API for Python
# https://developers.arcgis.com/python/

# this example shows an "arcpy" option
# The focus of this script is to create a unique value list


import arcpy

# a string for the park amenities layer
palayer = "Park Amenities"

# the string for the hosted park amenities feature service
pa_service = "https://services5.arcgis.com/54falWtcpty3V47Z/arcgis/rest/services/Public_Park_Amenities/FeatureServer/0"

# Make a feature layer from the service
arcpy.MakeFeatureLayer_management(pa_service, palayer)

# create an empty python list
pa_list = []


# Use a search cursor to iterate over all of the park amenities features
# Focus on only the "Amentity" data field
with arcpy.da.SearchCursor(palayer, 'Amenity') as srows:
    for srow in srows:

        # if the value is not in the list, then
        # append it to the list
        if srow[0] not in pa_list:
            pa_list.append(srow[0])

print (pa_list)

# Do something with the list

# Loop over the list to generate a query and print the count of features

for pa in pa_list:

    # Account for the unique value with an "apostrophe" s in Famer's
    # Triple double quotes use to ensure the proper syntax is used
    # NOTE:  the "double" single quote....this is the same syntax
    # that is produced when manually performing this query in Pro
    if pa == "Farmer's Market Area":
        pa = """Farmer''s Market Area"""
        print (pa)

    # create the query variable to use in the Make Feature Layer
    query = """"Amenity" = '""" + pa + """'"""
    print (query)

    # perform clean up if the feature layer already exists
    # delete it if it does
    if arcpy.Exists(palayer):
        arcpy.Delete_management(palayer)

    # Create the feature layer using the query
    # similar to the "Definition Query" in Pro
    arcpy.MakeFeatureLayer_management(pa_service, palayer, query)

    # Get a count of queried features
    result = arcpy.GetCount_management(palayer)
    print ("Feature Count of " + pa + " amenities is: " + str(result))
