# Amador Winery Siting Script 

# Created by:  Kent Heiner
#              www.kaheiner@ucdavis.edu
# Created on:  2018.04.08
# Updated on:  2018.05.11


'''
This script will perform various geoprocessing routines for specified
feature classes within Amador County to help locate available AG zoned
parcels for potential winery purchase.

This script demonstrates:

1. Workspaces
2. Input/output feature classes
3. Use of ArGIS geoprocessing functions
4. Defining parameters for the ArcGIS Buffer routines programmatically
5. try: except: blocks for error handling
6. traceback module for additional error handling
7. os.path.join routine for different workspace/outpath .gdb
8. Copy features to new feature classes
9. Creating layers for attribute selection and query functions
        
'''

print 'Amador Winery Siting Script\n'

# 1. import arcpy module

import arcpy, os, sys, traceback

# 2. Create workspace                       

arcpy.env.workspace = 'F:\\GEOG 375_Intro Python Prog_Jennings\\Project\\Data\\AmadorCounty.gdb\\'

# Add a variable set to the output location of the file geodatabase
# NOTE: not a workspace

outpath = 'F:\\GEOG 375_Intro Python Prog_Jennings\\Project\\MyData\\WinerySiting.gdb'

# 3. Create variable definitions for geoprocessing functions

infile = 'Zoning'
out_ag_fc = 'AgZoneOnly'
out_ag_fc_attrib = 'out_ag_attrib.dbf'
out_par_ag_fc = 'ParcelAgZone'
out_plymouth_fc = 'PlymouthCityBoundary'
out_plym_buf_par_ag = 'Plymouth_4mi_Parcels'
CityBoundary = 'CityBoundary'
parcels = 'Parcels'
outbuffer = 'PlymouthCityBnd_4mi_Buf'
buffer_value = 4
buffer_units = 'Miles'
buffer_distance = str(buffer_value) + ' ' + buffer_units

# !!!! Add variables that combine the output path and the output fc/tables !!!!
fgd_join_fc = os.path.join(outpath, out_ag_fc )
fgd_join_dbf = os.path.join(outpath, out_ag_fc_attrib)
fgd_join_par_ag = os.path.join(outpath, out_par_ag_fc)
fgd_join_plym = os.path.join(outpath, out_plymouth_fc)
fgd_join_outbuffer = os.path.join(outpath, outbuffer)
fgd_join_plym_buf_par_ag = os.path.join(outpath, out_plym_buf_par_ag)

#3a. Create variables for layers
agzone_layer = 'AG_Zone_AmC'
zoning_layer = 'Zoning_AmC'
parcel_layer = 'Parcels_AmC'
ag_par_layer = 'ParcelAgZone_AmC'
city_bnd_layer = 'City_Bnd_AmC'
plymouth_buf_layer = 'Plymouth_4mi_Buf'


# 4.  Add try: and except: blocks

try:


#Check to see if the feature layer exists; if so, delete it 

    if arcpy.Exists(zoning_layer):
        arcpy.Delete_management(zoning_layer)

    # Make the feature layer for Zoning
    arcpy.MakeFeatureLayer_management(infile, zoning_layer) 
    print "Making feature layer " + zoning_layer
    # 5. Create a query statement to select AG Zone 

    # Create query statements

    query = """"Zoning" = 'AG'"""
    query2 = """"NAME" = 'Plymouth'"""

    print "Query performed to isolate agriculturally zoned areas of Amador County"
    
    # 6. Select by attributes using the query

    # Select features by attribute using query
    arcpy.SelectLayerByAttribute_management(zoning_layer, "NEW_SELECTION", query)
    
    # 7. Copy the selected AG Zoning features to a new shapefile

    if arcpy.Exists(fgd_join_fc):
        arcpy.Delete_management(fgd_join_fc)
        
    arcpy.CopyFeatures_management(zoning_layer,fgd_join_fc)
    print "Copied selected features from " + zoning_layer  + " to: \n" + fgd_join_fc
    
    # print the number of records in the feature layer with query
    result = arcpy.GetCount_management(zoning_layer)

    
    
    print "Number of AG Zone polygons in Amador County " + zoning_layer +" : " + str(result)
    

    # 8. Select Parcels that intersect AG Zone  
    
    # Check to see if the feature layer exists; if so, delete it
    if arcpy.Exists(fgd_join_par_ag):
        arcpy.Delete_management(fgd_join_par_ag)

    # Make the feature layer for parcels
    arcpy.MakeFeatureLayer_management(parcels, parcel_layer)
    print "Making feature layer " + parcel_layer

    # Make the feature layer for Ag Zone Only
    arcpy.MakeFeatureLayer_management(fgd_join_fc, agzone_layer)
    print "Making feature layer " + agzone_layer
    
    # 9)a. Intersect AG Zone and Parcels
    arcpy.SelectLayerByLocation_management(parcel_layer, "HAVE_THEIR_CENTER_IN", agzone_layer)
    print "Intersecting features within " + parcel_layer  + " and " + agzone_layer
    
    arcpy.CopyFeatures_management(parcel_layer,fgd_join_par_ag)
    print "Copied selected features from " + parcel_layer  + " to: \n" + fgd_join_par_ag
    
    # print the number of records in the feature layer with query
    result2 = arcpy.GetCount_management(parcel_layer)
    print "Number of AG Zone/Parcel polygons in Amador County " + parcel_layer +" : " + str(result2)

    
    ## Remove layer if exists
    if arcpy.Exists(city_bnd_layer):
        arcpy.Delete_management(city_bnd_layer)

    # Make the feature layer for City Boundary
    arcpy.MakeFeatureLayer_management(CityBoundary, city_bnd_layer) 
    print "Making feature layer " + city_bnd_layer
    
    # Select features by attribute using query2
    arcpy.SelectLayerByAttribute_management(city_bnd_layer, "NEW_SELECTION", query2)
    print "Selecting city boundary of Plymouth only"

    # 9)b. Copy the selected Plymouth features to a new shapefile

    if arcpy.Exists(fgd_join_plym):
        arcpy.Delete_management(fgd_join_plym)

    arcpy.CopyFeatures_management(city_bnd_layer,fgd_join_plym)
    print "Copied selected features from " + city_bnd_layer  + " to: \n" + fgd_join_plym

    # print the number of records in the feature layer with query2
    result3 = arcpy.GetCount_management(city_bnd_layer)
    print "Number of City Boundary polygons for Plymouth " + city_bnd_layer +" : " + str(result3)

    
    # 10. Add Buffer routine 

    if arcpy.Exists(fgd_join_outbuffer):
        arcpy.Delete_management(fgd_join_outbuffer)

    print "Starting Buffer routine"

    arcpy.Buffer_analysis(fgd_join_plym, fgd_join_outbuffer, buffer_distance)

    print "Finished Buffer routine"
    
    # Make the feature layer for Ag Parcel
    if arcpy.Exists(ag_par_layer):
        arcpy.Delete_management(ag_par_layer)

    
    arcpy.MakeFeatureLayer_management(fgd_join_par_ag, ag_par_layer) 
    print "Making feature layer " + ag_par_layer
    
    # Make the feature layer for Plymouth Buffer
    if arcpy.Exists(plymouth_buf_layer):
        arcpy.Delete_management(plymouth_buf_layer)

    
    arcpy.MakeFeatureLayer_management(fgd_join_outbuffer, plymouth_buf_layer) 
    print "Making feature layer " + plymouth_buf_layer

    
    #Intersect Plymouth Buffer with Ag Parcels

    arcpy.SelectLayerByLocation_management(ag_par_layer, "INTERSECT",
                                           plymouth_buf_layer) 
    print "Intersecting features within " + ag_par_layer + " and " + plymouth_buf_layer
        
    # 11. Copy the selected Plymouth buffer/parcel features to a new shapefile

    if arcpy.Exists(fgd_join_plym_buf_par_ag):
        arcpy.Delete_management(fgd_join_plym_buf_par_ag)

    arcpy.CopyFeatures_management(ag_par_layer, fgd_join_plym_buf_par_ag)
    print "Copied selected features from " + ag_par_layer + " to: \n" + fgd_join_plym_buf_par_ag
    
    # print the number of records in the feature layer
    result4 = arcpy.GetCount_management(ag_par_layer)
    print "Number of parcels within 4mi Plymouth buffer zone for " + ag_par_layer +" : " + str(result4)

    print "Completed Script"

# 12. Add exception code within except block

except:

    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000
    
    # Notice the indent because of the except statement
    
        tb = sys.exc_info()[2]
        tbinfo = traceback.format_tb(tb)[0]
        pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
        msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

        # prints error messages in the Progress dialog box
        arcpy.AddError(msgs)   
        arcpy.AddError(pymsg)

        # prints messages to the Python Shell
        print msgs
        print pymsg

        arcpy.AddMessage(arcpy.GetMessages(1))

        print arcpy.GetMessages(1)

