# Demo 9b - Mapping Module Methods

# Created by:  Nathan Jennings
#              www.urbandalespatial.com
# Created on:  03.13.2011
# Updated on:  12.14.2014
# Copyright:   2011

'''
This script provides an overview for some of the methods associated with the data frame,
layers, and layout elements.  The Map Document methods are implemented in the Mapping Module
Overview.py script.  The script processes different methods, properties, and functions, and
then exports the changed maps to separate PDF documents.  Six maps are written in this script.

NOTE:  The user may need to change the path locations to data and map output to use
this script.

Methods demonstrated (and associated class):

zoomToSelectedFeatures() - data frame
getSelectedExtent() - layer
saveACopy() - layer

Additional Properties demonstrated (and associated class):

scale - data frame
autoAdd - legend element
showLabels - layer

Additional functions demonstrated:

AddLayer


'''
print 'Demo 9b - Mapping Module Methods\n'

import arcpy, sys, traceback, datetime
from arcpy.mapping import *   # Eliminates the need to write arcpy.mapping
                              # before each mapping module related class, function
                              # or method

author = 'N. Jennings'

# reformat date to MM.DD.YYYY
# NOTE: using a lowercase %y will result in MM.DD.YY format
# See Python.org or text regarding the datetime module

CUR_DATE = datetime.date.today().strftime('%m.%d.%Y')

try:

    # Get the map document.
    # In this case a custom template set up for map production
    # change the paths as needed to run the script form a local system
    datapath = 'C:\\PythonPrimer\\Chapter09\\'
    mappath = datapath + 'MyData\\Maps\\'    # output for PDFs
    mxd = MapDocument(datapath + 'Mapping_Module_Overview.mxd')

    # Get a list of data frames
    # [0] inicates the first (or only data frame, in this case)
    # The data frame name is "Layers", the default name for the
    # primary data frame
    dataframe = ListDataFrames(mxd, "Layers") [0]

    
    # Get a list of layers in the table of contents of the map document
    TOCLayers = ListLayers(mxd)

    # Get a list of layout elements (text elements)
    tElements = ListLayoutElements(mxd, "TEXT_ELEMENT")

    #for tElement in tElements:

        # if the text element name is 'Map Title',
        # then assign a specific name for the title
        # see the elements properties in ArcMap (under size and position)
        
        #if tElement.name == 'Map Title':
            
            #tElement.text = 'Test Map'

    print 'Processing layout elements...'
    
    # loop through the layers
    for TOCLayer in TOCLayers:
        #print 'Layer Name: ' + str(TOCLayer.name)
        
        #print 'Longname: ' + str(TOCLayer.longName)  # longName provides
                                                    # the ability to use
                                                    # "Group Layers"

        # Detail Map is the group layer
        # Neighborhoods is a layer within the group
        
        if TOCLayer.longName == 'Detail Map\Neighborhoods':

            # perform a Select by Attriute to select a single neighborhood

            query = """"NAME" = 'Alkali Flat'"""
            print query

            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query)

            result = arcpy.GetCount_management(TOCLayer)
            print 'Number of selected features: ' + str(result)

            # zoom to the extent of the selected feature
            
            dataframe.zoomToSelectedFeatures()
            
            #tElements = ListLayoutElements(mxd, "TEXT_ELEMENT")

            # cycle through the text elements to change the map title and date
            for tElement in tElements:

                # if the text element name is 'Map Title',
                # then assign a specific name for the title
                # see the elements properties in ArcMap (under size and position)
                
                if tElement.name == 'Map Title':
                    
                    tElement.text = 'Alkali Flat'

                if tElement.name == 'Print Date':  

                    tElement.text = str(CUR_DATE)  # assign the current date
                                                   # to this text element

            print 'Exporting Map 1...'
            if arcpy.Exists(mappath + 'ZoomToSelectedFeatures_Map1.pdf'):
                arcpy.Delete_management(mappath + 'ZoomToSelectedFeatures_Map1.pdf')
            
            ExportToPDF(mxd, mappath + 'ZoomToSelectedFeatures_Map1.pdf')

            # Map 2
            # clear any selected features        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")

            
            dataframe.zoomToSelectedFeatures()

            print 'Exporting Map 2...'
            if arcpy.Exists(mappath + 'NoSelectedFeaturesZoom_Map2.pdf'):
                arcpy.Delete_management(mappath + 'NoSelectedFeaturesZoom_Map2.pdf')

            ExportToPDF(mxd, mappath + 'NoSelectedFeaturesZoom_Map2.pdf')

            # Map 3
            query = """"NAME" = 'Downtown'"""
            print query

            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query)

            dataframe.extent = TOCLayer.getSelectedExtent()
            
            #tElements = ListLayoutElements(mxd, "TEXT_ELEMENT")
            
            # cycle through the text elements to change the map title and date
            # do this again because the map changed neighborhoods
            for tElement in tElements:

                # if the text element name is 'Map Title',
                # then assign a specific name for the title
                # see the elements properties in ArcMap (under size and position)
                
                if tElement.name == 'Map Title':
                    
                    tElement.text = 'Downtown'

            print 'Exporting Map 3...'
            if arcpy.Exists(mappath + 'GetSelectedExtentfromLayer_Map3.pdf'):
                arcpy.Delete_management(mappath + 'GetSelectedExtentfromLayer_Map3.pdf')
            
            ExportToPDF(mxd, mappath + 'GetSelectedExtentfromLayer_Map3.pdf')

            # Clear feature selection to remove the "highlighted features"

            # Map 4
            # clear any selected features, but don't change the zoom extent        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")

            print 'Exporting Map 4...'
            if arcpy.Exists(mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf'):
                arcpy.Delete_management(mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf')
            
            ExportToPDF(mxd, mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf')

            # Map 5
            TOCLayer.definitionQuery = query     # Assign the existing query to the
                                                 # definitionQuery property of the layer
            

            dataframe.scale = dataframe.scale * 1.1   # Add a 10% "buffer" to the extent
                                                      # so that the edges of the feature
                                                      # are not touching the map frame

            # The exported map will contain the definition query
            # the modified data frame scale
            print 'Exporting Map 5...'
            if arcpy.Exists(mappath + 'DefinitionQuery_ScaleChange_Map5.pdf'):
                arcpy.Delete_management(mappath + 'DefinitionQuery_ScaleChange_Map5.pdf')
            
            ExportToPDF(mxd, mappath + 'DefinitionQuery_ScaleChange_Map5.pdf')

            # Implement a definition query
            # get the extent of the features (in this case the extent of
            # the queried features
            # export the map to a PDF file

            # Map 6
            query = """"NAME" = 'Newton Booth'"""

            TOCLayer.definitionQuery = query   # Assign the existing query to the
                                               # definitionQuery property of the layer

            # get the extent of the features (in this case the definitin queried features
            # assign this extent to the data frame extent
            dataframe.extent = TOCLayer.getExtent()
            dataframe.scale = dataframe.scale * 1.1  # Add a 10% buffer to the extent

            for tElement in tElements:

                # if the text element name is 'Map Title',
                # then assign a specific name for the title
                # see the elements properties in ArcMap (under size and position)
                
                if tElement.name == 'Map Title':
                    
                    tElement.text = 'Newton Booth'

            # The exported map will contain the definition query
            # the modified data frame scale
            print 'Exporting Map 6...'
            if arcpy.Exists(mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf'):
                arcpy.Delete_management(mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf')
            
            ExportToPDF(mxd, mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf')
            
            # Map 7
            # Add a new layer to the map

            # Get the city facilities.lyr file from disk
            # Do not "Auto Add" the new layer to the legend
            # Add the layer to the map frame
            # Turn labels on for the city facilities layer based on the predefined
            # label attrbitue in the .lyr file
            # Save the new layer changes (labels turned on) to a new layer (.lyr) file
            # Export a new map

            legend = ListLayoutElements(mxd, "LEGEND_ELEMENT") [0] # Get the first and only legend


            cityFacilitiesLayer = Layer(datapath + 'data\\city facilities.lyr')
            legend.autoAdd = False  # set the autoAdd before adding the layer
            AddLayer(dataframe, cityFacilitiesLayer)   # Add the new layer

            # new list contains the new city facilities layer
            NewLayerList = ListLayers(mxd)
            
            for lyr in NewLayerList:

                # check to see if the new layer name is present
                if lyr.name == 'City Facilities':

                    # if so, turn the labels on
                    lyr.showLabels = True
                    lyr.saveACopy(mappath + 'City_Facilities_Labels.lyr')
            

            # Save the layer file to a new .lyr file name
            # The layer saveACopy only works for any layer that is a .lyr file
            # save and saveACopy cannot be used with other layers in the Table of Contents
            # that to not originate from a .lyr file
            #arcpy.RefreshActiveView()
            # ExportToPDF(mxd, mappath + 'City_Facilities_Map71.pdf')

            print 'Exporting Map 7...'
            if arcpy.Exists(mappath + 'City_Facilities_Map7.pdf'):
                arcpy.Delete_management(mappath + 'City_Facilities_Map7.pdf')
            
            ExportToPDF(mxd, mappath + 'City_Facilities_Map7.pdf')

            
                
    print 'Completed Map Updates'

except:
    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000

    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print msgs
    print pymsg
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print arcpy.GetMessages(1)
