# Demo 9a - Mapping Module Overview

# Created by:  Nathan Jennings
#              www.urbandalespatial.com
# Created on:  02.20.2011
# Updated on:  12.14.2014
# Copyright:   2011

'''
This script provides an overview for some of the properties for different map
elements used with the mapping module.  Some of these properties are printed
out for reference, while others are used to modify the look and feel of the map
(e.g. turning a layer on/off, changing the transparency.

The user can modify these values to learn more about how changes in these
properties affect the output map.

The output of this script is a PDF document.

NOTE:  The PDF will be overwritten with each implementation of this script.

Rename the existing file before implementing the script more than one time.


'''
print 'Demo 9a - Mapping Module Overview\n'

import arcpy, os, sys, traceback, datetime
from arcpy.mapping import *   # Eliminates the need to write arcpy.mapping
                              # before each mapping module related class, function
                              # or method

author = 'N. Jennings'

# reformat date to MM.DD.YYYY
# NOTE: using a lowercase %y will result in MM.DD.YY format
# See Python.org or text regarding the datetime module

CUR_DATE = datetime.date.today().strftime('%m.%d.%Y')

try:

    # Get the map document.
    # In this case a custom template set up for map production
    # change the paths as needed to run the script form a local system
    datapath = 'C:\\PythonPrimer\\Chapter09\\'
    mappath = datapath + 'MyData\\Maps\\'    # output for PDFs
    mxd = MapDocument(datapath + 'Mapping_Module_Overview.mxd')

    # Report some of the Map Document Properties

    print 'Map Document Title: ' + str(mxd.title)
    print 'Map Document Author: ' + str(mxd.author)
    print 'Map Document Summary: ' + str(mxd.summary)
    print 'Map Document Description: ' + str(mxd.description)
    print 'Map Document last Date Saved: ' + str(mxd.dateSaved)
    print 'Is the Map Document Relative Path checked? : ' + str(mxd.relativePaths)

    # Get a list of data frames
    # [0] inicates the first (or only data frame, in this case)
    # The data frame name is "Layers", the default name for the
    # primary data frame
    dataframe = ListDataFrames(mxd, "Layers") [0]

    # Report some properties about the map frame named "Layers"

    print 'Map Frame Map Units: ' + str(dataframe.mapUnits)
    print 'Map Frame Scale: ' + str(dataframe.scale)

    # See ArcGIS Help under 'spatialreference' (all one word) for more information
    print 'Map Spatial Reference: ' + str(dataframe.spatialReference.name)
    print 'Map Frame Anchor Point X Position (page units): ' + str(dataframe.elementPositionX)
    print 'Map Frame Anchor Point Y Position (page units): ' + str(dataframe.elementPositionY)
    print 'Map Frame Width (page units): ' + str(dataframe.elementWidth)
    print 'Map Frame Height (page units): ' + str(dataframe.elementHeight)

    # Get the extent of the data frame and assign it to a variable
    mapExtent = dataframe.extent

    # The following syntax prints out the Min and Max X and Y values
    # for the map extent of the data frame.
    # Search on Extent in the ArcGIS Help to find more details on
    # the extents XMin, YMin, XMax, and YMax

    # The string formatting can be found in a Python text or at Python.org
    # %f indicates a decimal value will be added to a string
    # each %f below is substituted with the appropriate mapExtent.<value>
    
    print 'Map Extent of Data Frame is: \n' + \
          'XMin: %f, YMin: %f, \nXMax: %f, YMax: %f' % \
             (mapExtent.XMin, mapExtent.YMin, mapExtent.XMax, mapExtent.YMax)

    
    # Get a list of layers in the table of contents of the map document
    TOCLayers = ListLayers(mxd) 

    print 'Processing layout elements...'
    
    # loop through the layers
    for TOCLayer in TOCLayers:
        print 'Layer Name: ' + str(TOCLayer.name)
        
        print 'Longname: ' + str(TOCLayer.longName)  # longName provides
                                                    # the ability to use
                                                    # "Group Layers"

        # Detail Map is the group layer
        # Neighborhoods is a layer within the group
        
        if TOCLayer.longName == 'Detail Map\Neighborhoods':

            TOCLayer.transparency = 50  # 50% transparency
            
        if TOCLayer.longName == 'Detail Map\Streets':

            TOCLayer.showLabels = True  # turn labels on

        if TOCLayer.longName == 'Detail Map\Parcels':

            TOCLayer.visibility = False  # turn layer off
    
    # Get a list of "text" type layout elements from the map document
    tElements = ListLayoutElements(mxd, "TEXT_ELEMENT")

    print 'Processing text elements...'
    for tElement in tElements:

        # if the text element name is 'Map Title',
        # then assign a specific name for the title
        # see the elements properties in ArcMap (under size and position)
        
        if tElement.name == 'Map Title':
            
            tElement.text = 'Test Map'
            tElement.elementPositionX = 8.7  # Anchor point is lower left
                                             # The anchor point is used
                                             # to locate the lower left
                                             # page position of the text

            # The X and Y positions (and Height and Width
            # can be found in the element properties
            # of the layout object under Size and Position

        if tElement.name == 'Print Date':  

            tElement.text = str(CUR_DATE)  # assign the current date
                                           # to this text element


    # Check to see if PDF exists, if it does, delete it
    if arcpy.Exists(mappath + 'test_map.pdf'):
        arcpy.Delete_management(mappath + 'test_map.pdf')
        
    print 'Writing PDF file...'
    
    # Create the PDF document
    ExportToPDF(mxd, mappath + 'test_map.pdf')
    print 'Created : ' + 'test_map.pdf'

    # Alternatively, the map can be printed to a local printer
    # This is commented out and can be changed by the code developer

    # PrintMap(mxd)  # prints map to default printer
    # PrintMap(mxd, '\\\\network_location\\printer_name') # print to networked printer


    # Save Changes
    # Commented out so that the user can perform this operation if desired.

    # mxd.save() # performs the same operation as File--Save in ArcMap
               # if this is a map template that is used as the basis for
               # map production, saving the map may not be warranted
               # this will save any changes made in the script

    # mxd.saveACopy(<path and file name of different MXD>, '10.0')

    # performs the same operation as File--SaveACopy in ArcMap
    # in this case an ArcGIS 9.3 version

    # if this is a map template that is used as the basis for
    # map production, saving the map may not be warranted
    # default is ArcGIS version 10, if no parameter is assigned
    
    print 'Completed Map Updates'

except:
    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print msgs
    print pymsg
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print arcpy.GetMessages(1)
