# Demo 5b  Select Layer by Attribute and Select Layer by Location (using keyword arguments)
# Created by:  Nathan Jennings
#              www.urbandalespatial.com
# Created on:  01.16.2011
# Updated on:  02.03.2020
# Copyright:   2015

'''
Example of using the Make Feature Layer tool that is used to implement the
Select Layer by Attribute and Select Layer by Location ArcGIS tools.  The
Copy Features and Copy Row tools are used to store the selected records.
Copy Features will store the selected geographic features and attributes from
the parcel file, whereas, the Copy Rows will be used to store the selected
attributes from the streets file.

'''
print ('Demo 5b - Select Layer by Attribte and Location\n')

import arcpy, sys, traceback

#arcpy.env.workspace = "C:\\PythonPrimer\\Chapter05\\Data\\"
arcpy.env.workspace = "C:\\temp\\Chapter05\\Data\\"

streets = "Sacramento_Streets.shp"
street_layer = "street_layer"

parcels = "City_Parcels.shp"
parcel_layer = "parcel_layer"
search_dist = '200 FEET'

out_parcel_fc = "out_parcels.shp"
out_street_attributes = "out_street_attributes.dbf"




try:

    # 1. Make a feature layer for streets

    if arcpy.Exists(street_layer):
        arcpy.Delete_management(street_layer)
    
    arcpy.MakeFeatureLayer_management(streets, street_layer)

    # 2. Create a query statement to select highways

    # Create query statement
    query = """"CLASS" = 'H'"""

    # 3. Select by attributes using the query

    # Select features by attribute using query

    # original SelectLayerByAttribute syntax
    #arcpy.SelectLayerByAttribute_management(street_layer,"NEW_SELECTION", query)

    print ("using keywords in SelectLayerByAttribute...")
    print ("Since no selection type is provided, the routine will use NEW_SELECTION")
    arcpy.SelectLayerByAttribute_management(street_layer, where_clause=query)

    result = arcpy.GetCount_management(street_layer) 
    print ("Number of selected streets in the street layer " + street_layer + " with a query: "+ str(result))

    
    

    # 4. Select parcels that are 200 ft from highways

    # 4a. Make feature layer

    if arcpy.Exists(parcel_layer):
        arcpy.Delete_management(parcel_layer)
        
    arcpy.MakeFeatureLayer_management(parcels, parcel_layer)

    
    # 4b. Select Parcels

    # This is the original syntax for the SelectLayerByLocation routine
    #arcpy.SelectLayerByLocation_management(parcel_layer, "INTERSECT", \
    #                                       street_layer, search_distance, "NEW_SELECTION") 


    #Here it is with keyword arguments.  Notice the order is changed, too.
    # Notice also the selection type is not defined, so this will be a "NEW_SELECTION"
    arcpy.SelectLayerByLocation_management(parcel_layer, select_features = street_layer, \
                                           search_distance = search_dist, overlap_type = "INTERSECT")

    # 4c. Get the number of selected records and create a print statement for each
    
    # print the number of selected records in the respective layers with a query
    result = arcpy.GetCount_management(street_layer) 
    print ("Number of selected streets in the street layer " + street_layer + " with a query: "+ str(result))

    result = arcpy.GetCount_management(parcel_layer) 
    print ("Number of selected parcels in the parcel layer " + parcel_layer + \
          "\n within a distance of " + search_dist + ": "+ str(result))

    # 5. Write out selected parcel features to file (SHP)

    # Copy the selected polygon features to a new shapefile

    if arcpy.Exists(out_parcel_fc):
        arcpy.Delete_management(out_parcel_fc)
        
    arcpy.CopyFeatures_management(parcel_layer, out_parcel_fc)
    print ("Copied selected features from " + parcels  + " to " + out_parcel_fc)

    # 6. Write out selected street attributes to a table file (DBF)


    # Copy the selected street segment attributes to a dBase table

    if arcpy.Exists(out_street_attributes):
        arcpy.Delete_management(out_street_attributes)
        
    arcpy.CopyRows_management(street_layer, out_street_attributes)
    print ("Copied selected attributes from " + streets + " to " + out_street_attributes)

    print ("Completed Script")


except:
    # https://pro.arcgis.com/en/pro-app/arcpy/get-started/error-handling-with-python.htm
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" + str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
