# Final Project Script to create direction field from COGO data field direction

# Created by: Gary Yuki
# Created on 04.09.2017
# Updated on 

'''
This script attempts to take the information created from the ArcGIS COGO Traverse tool
and create additional fields that can be used as labels by adding special characters
for degrees/minutes/seconds and additionally add a label name for segments that
can not be labled with direction and distance when the segments are too small to fit
the text. The small segment labels would be L1, L2, etc. for line segments and C1,C2,
etc. for curve segments. The user would need to add what length the segments are
to be labeled too small and need to be called out.

Three tables will be constructed.
1) The "insert_label_table" will hold all of the newly formatted data fields for
labeling the line segments
2) The "curve_callout_table" will compile only the small or "Called out" curves to
insert as a reference table in the drawing.
3) The "line_callout_table" will compile only the small or "Called out" Line segments
to insert as a reference table with in the drawing.

To label drawing with the "insert_label_table, the table must be joined with the  COGO shapefile
in arcMap.  In the label tab add an following expression to label with a call out or a standard label -

def FindLabel ( [COGO_Boundary.Distance], [insert_label_table.Label_Segment] , [insert_label_table.Label_Direction]  , [insert_label_table.Label_Distance] ):
  if ( [COGO_Boundary.Distance]) <300:
    return [insert_label_table.Label_Segment] 
  else:
    return [insert_label_table.Label_Direction] + '\n' + [insert_label_table.Label_Distance] 

'''

import arcpy, sys, traceback, re, os

# Set the current workspace
arcpy.env.workspace = 'H:\\FinalProject\\FinalProject.gdb'
fgdpath = 'H:\\FinalProject\\FinalProject.gdb\\' #directory path to .gdb
cogo_shp = 'COGO_Boundary.shp' #original data from COGO
#metes_fgd = fgdpath + 'metes_bounds_table' #table with updated label info



#assign a variable to hold the label output table
label_table = 'insert_label_table'
label_table_fgd = fgdpath + label_table
curve_table = 'curve_callout_table'
curve_table_fgd = fgdpath + curve_table
line_table = 'line_callout_table'
line_table_fgd = fgdpath + line_table

#define special symbol variables
deg_sym = u"\u00b0"
min_sym = """'"""
sec_sym = '"'
ft_sym = " ft"
dim_length = 300  #user will need to define the maximum line lenght for callout
curveNum = 1 #label counter initialization
lineNum = 1 #label counter init

try:
    #create a table to store labels
    #check to see if the label_table exists to delete it
    if arcpy.Exists(label_table):
        arcpy.Delete_management(label_table)

    #create a new label_table
    arcpy.CreateTable_management(fgdpath,label_table)

    #add fields to the Label table
    arcpy.AddField_management(label_table, 'Label_FID', 'DOUBLE')
    arcpy.AddField_management(label_table, 'Label_Segment', 'TEXT', '', '', 20) 
    arcpy.AddField_management(label_table, 'Label_Distance', 'TEXT', '', '', 50) 
    arcpy.AddField_management(label_table, 'Label_Direction', 'TEXT', '', '', 50)

    #create a table to store curve segment callout
    #check to see if the curve_table exists to delete it
    if arcpy.Exists(curve_table):
        arcpy.Delete_management(curve_table)

    #create a new curve callout table
    arcpy.CreateTable_management(fgdpath,curve_table)

    #add fields to curve callout table
    arcpy.AddField_management(curve_table, 'Curve_Callout', 'TEXT', '', '', 20)
    arcpy.AddField_management(curve_table, 'Cord_Bearing', 'TEXT', '', '', 50)
    arcpy.AddField_management(curve_table, 'Chord_Length', 'TEXT', '', '', 50)
    arcpy.AddField_management(curve_table, 'Radius', 'TEXT', '', '', 20)
    arcpy.AddField_management(curve_table, 'Turn', 'TEXT', '', '', 20)


    #create a table to store label segment callout
    #check to see if the label_table exists to delete it
    if arcpy.Exists(line_table):
        arcpy.Delete_management(line_table)

    #create a new curve callout table
    arcpy.CreateTable_management(fgdpath,line_table)

    #add fields to curve callout table
    arcpy.AddField_management(line_table, 'Line_Callout', 'TEXT', '', '', 20)
    arcpy.AddField_management(line_table, 'Bearing', 'TEXT', '', '', 50)
    arcpy.AddField_management(line_table, 'Length', 'TEXT', '', '', 50)
    
    print 'Created tables and added fields'

    #setup search cursor to grab the direction data for updating with special characters
    field_list = ["Direction", "Distance", "Radius", "Side" ]
  

    with arcpy.da.SearchCursor(cogo_shp, field_list,) as srows:

        objID = 1
        
        for srow in srows:
                                    
            Direction = srow[0]
            Distance = srow[1]
            Radius = srow[2]
            Turn = srow[3]

            #split Direction into its components for later concatenation Direction looks like 'N 4-23-34 W'
            #The following will delimit the direction into 5 peices
            NS,Deg,Min,Sec,EW = re.split('[-  ]', Direction) 

            #Concatenate direction and special symbols to the Direction label
            label_dir = NS + " " + Deg + deg_sym + Min + min_sym + Sec + sec_sym + " " + EW
            label_dist = str(Distance) + ft_sym
            label_radi = str(Radius) + ft_sym
            label_turn = str(Turn)
            
            # Build short line segment lable callouts
            #convert Distance text to float to use expression
            dist_num = float(Distance)
            #convert Radius to integer, some fields have nulls
            radi_string = "0" + Radius
            radi_num = int(radi_string)
        
            if radi_num > 0: #Selects Curve line segment callout
                label_seg = str("C" + str(curveNum))

                # Update Curve Callout Table
                update_curve_call = ("Curve_Callout", "Cord_Bearing", "Chord_Length", "Radius", "Turn")
                with arcpy.da.InsertCursor(curve_table_fgd, update_curve_call,) as crow:
                        crow.insertRow((label_seg, label_dir, label_dist, label_radi, label_turn))

                #increment curve counter
                curveNum = curveNum + 1

            elif dist_num < dim_length :        #selects small line segment callout
                label_seg = str("L" + str(lineNum))

                #update line Callout Table
                update_line_call = ['Line_Callout', 'Bearing', 'Length']
                with arcpy.da.InsertCursor(line_table_fgd, update_line_call,) as lrow:
                        lrow.insertRow((label_seg, label_dir, label_dist))

                #increment line counter
                lineNum = lineNum + 1 

            else:
                label_seg = "0"

            #create a variable for the fields to update
            update_fields = ['Label_FID', 'Label_Segment', 'Label_Distance', 'Label_Direction']


            #create the update cursor to populate the label table
            #changing to an insert cursor
            with arcpy.da.InsertCursor(label_table_fgd, update_fields,)as irows:
                
                    irows.insertRow((objID, label_seg, label_dist, label_dir)) 


            objID += 1

    print "Finished updating rows in " + label_table
    print "Finished creating callout tables " + line_table + " and " + curve_table

except:

    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000
        
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print msgs
    print pymsg
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print arcpy.GetMessages(1)
        
