# Script written for creation of survey areas for Northern Goshawks using Model Builder and
# USDA Northern Goshawk Inventory and Monitoring Technical Guide specifications
# Available at: https://www.fs.fed.us/biology/wildecology/docs/GoshawkTechGuideJuly06.pdf
# ---------------------------------------------------------------------------
# NOGOscriptWithDissolve_export.py

# Created by: Emily Schultz
# Created on: 2017-04-08 19:48:30.00000
#   (generated by ArcGIS/ModelBuilder)

'''
Description: The script takes the Forest Service vegetation layer and selects out those areas
preferred by Northern Goshawks (WHRSIZE of 4, 5, or 6 in WHRDENSITY of M or D - tree size and
habitat types). Areas with habitat greater than 10 acres are of particular interest, so these
are selected for after acreages are calculated, and the other areas dropped. From there, layers
are removed as appropriate (in the following script, high severity burn in King Fire area, private
property, and slope greater than 35%), and then trimmed to the project area (Quidazoic Timber Sale
in this script). This file is then merged with goshawk PACs (Protected Activity Centers) found within
the project area, and the "Make Sample Point Grid" Tool is used to place points within these areas,
at a distance of 200 meters apart (x-axis), and a distance of 100 meters on the y-axis, with an offset
used for maximum survey coverage.

Make Sample Point Grid originally created by Carl Beyerhelm (GIS, GPS, and Data Management) 13 June
    2007. Contact information: Coconino National Forest, cbeyerhelm@fs.fed.us, 928.527.3635.
    Tool was updated by Michael Rich (USFS Region 6 DRM Geotools) 24 May 2010. Contact info:
    mrich@fs.fed.us, 541.383.4015

Model Builder was used to build the general body of this tool, and then modifications were made
as necessary using IDLE.

'''

print 'Northern goshawk survey location determination commencing...'

# Import arcpy module
import arcpy, os, sys, traceback

arcpy.env.workspace = "C:\PythonPrimer\FinalProject\NOGOdata.gdb"

inpath = "C:\PythonPrimer\FinalProject\NOGOdata.gdb\\"
outpath = "C:\PythonPrimer\FinalProject\NOGOdata.gdb\OutputFiles\\"

# Load required toolboxes
# Did not end up using this tool - script old
#arcpy.ImportToolbox("C:\PythonPrimer\FinalProject\MakeSamplePointGrid.tbx", "SamplePointGrid")

# Local variables:

# Vegetation Files
vegetation = "vegetation" #this is the original Forest Service vegetation file
veg456md_layer = "veg456md_layer" #this is the layer created using the selection of attributes
veg456md_fc = outpath + "veg456md"
veg_dissolve = outpath + "veg_dissolve" #dissolved vegetation layer for acreage calculation
veg_eraseBurn = outpath + "veg_eraseBurn" #vegetation layer minus areas of high severity burn
vegAcre_layer = "vegAcre_layer" #layer with acreages included for vegetation polygons
veg_nogoClip = outpath + "veg_Clip" # "final" vegetation layer clipped to project area

# Goshawk PACs
NOGOpacs = "NOGOpacs" #goshawk Protected Activity Centers (PACs) within the Eldorado National Forest (ENF)
nogoQuid_layer = "nogoQuid_layer" #goshawk PACs that fall within project area
nogo_buffer = "nogo_buffer"

# Veg plus nogo files various outputs
vegNogo = outpath + "vegNogo" #vegetation layer clipped to project survey areas
vegNogo_erasePrivate = outpath + "vegNogo_erasePrivate" #project survey area with private land ownership erased
vegNogo_eraseSlope = outpath + "vegNogo_eraseSlope" #project survey area with slope >35% erased

# Timber Sale Units
Quidazoic_Units = "Quidazoic_Units"
quid_buffer = outpath + "quid_Buffer" #timber units with a quarter mile buffer for complete survey coverage

# Erase Files
burn_sev = "burn_severity_King" #file showing areas with no change, low, medium, and high burn severities
kingHigh_layer = "kingHigh_layer" #layer depicting areas of high burn severity only
ownership = "ownership" #area showing basic landownership on ENF
private_layer = "private_layer" #layer representing those areas that are non-Forest Service lands
Slope35 = "Slope35" #file with slope values 
slope_layer = "slope_layer" #layer representing those areas with >35% slope

# Final Output
nogoFishnet = "nogoFishnet" #output representing fishnet survey locations
nogoFishnet_label = "nogoFishnet_label" #fishnet output file
nogoFinalSurveyArea = "nogoFinalPoints"

try:

    # Process: Select Layer By Attribute - Selecting for vegetation with values of 4, 5, 6 paired with m or d.
    # This represents suitable habitat for goshawks 
    # Checking for previous versions of veg456md layer
    print 'Checking for previous versions of vegetation layer with 4D, 5D, 6D, 4M, 5M, and 6M vegetation types...'

    if arcpy.Exists(veg456md_layer):
        arcpy.Delete_management(veg456md_layer)
    
    # Creating feature layer and selecting for 4D, 5D, 6D, 4M, 5M, and 6M vegetation types

    print 'Making vegetation feature layer...'

    arcpy.MakeFeatureLayer_management(vegetation, veg456md_layer)

    print 'Selecting appropriate vegetation types for feature layer...'

    # numbers in this field are string values, and come up with quotes when using select by attribute tool
    query = """("WHRSIZE" = '4' AND "WHRDENSITY" = 'D') OR ("WHRSIZE" = '5' AND "WHRDENSITY" = 'D') OR ("WHRSIZE" = '6' AND "WHRDENSITY" = 'D') OR ("WHRSIZE" = '4' AND "WHRDENSITY" = 'M') OR ("WHRSIZE" = '5' AND "WHRDENSITY" = 'M') OR ("WHRSIZE" = '6' AND "WHRDENSITY" = 'M')"""    

    arcpy.SelectLayerByAttribute_management(veg456md_layer, "NEW_SELECTION", query)
    
    print "Vegetation layer created."

    #### Instead of creating a feature class here, consider a print statement showing the possible combinations selected for from 'WHRSIZE' and 'WHRDENSITY' ####

    # Creating a veg feature class with 4, 5, 6, m, d for reference
    print 'Creating vegetation feature class to reflect 4D, 5D, 6D, 4M, 5M, and 6M vegetation type selection...'

    if arcpy.Exists(veg456md_fc):
        arcpy.Delete_management(veg456md_fc)

    arcpy.CopyFeatures_management(veg456md_layer, veg456md_fc)

    print 'Copied features from ' + inpath + vegetation + ' to ' + veg456md_fc

    # Use cursor to show values selected in veg layer for goshawk survey area

    print 'Using cursor to confirm selection of 4D, 5D, 6D, 4M, 5M, and 6M types...'
    
    whrsize_field = "WHRSIZE"
    whrdensity_field = "WHRDENSITY"

    with arcpy.da.SearchCursor(veg456md_layer, (whrsize_field, whrdensity_field)) as srows:
        for srow in srows:
            whrsize_field = srow[0]
            whrdensity_field = srow[1]

            print 'The following vegetation type is included in the goshawk survey area: ' + str(whrsize_field) + str(whrdensity_field)

    # delete cursors
    del srow, srows  

    # Process: Dissolve
    # Checking for previous versions of layer
    print 'Checking for previous versions of dissolved vegetation feature class...'

    if arcpy.Exists(veg_dissolve):
        arcpy.Delete_management(veg_dissolve)
        
    # Dissolving layer selected for above so that accurate acreage calculations can be obtained
    print 'Dissolving veg456md_layer and preparing for area calculation...'

    arcpy.Dissolve_management(veg456md_layer, veg_dissolve, "", "", "SINGLE_PART", "DISSOLVE_LINES")
    
    # Process: Select Layer By Attribute - Selecting for areas of hihg burn severity from King Fire
    # Checking for previous versions of high burn severity layer

    print 'Checking for previous versions of high burn severity layer...'

    if arcpy.Exists(kingHigh_layer):
        arcpy.Delete_management(kingHigh_layer)
    
    # Creating feature layer and selecting for high burn severity
    print 'Making burn severity feature layer...'

    arcpy.MakeFeatureLayer_management(burn_sev, kingHigh_layer)

    query2 = """"BURNSEV_DESC" = 'High'"""

    arcpy.SelectLayerByAttribute_management(kingHigh_layer, "NEW_SELECTION", query2)

    # Process: Erasing high burn severity from dissolved veg layer
    # Checking for previous versions of veg minus high burn severity layer

    print 'Checking for previous versions of vegetation feature class with high burn severity removed...'

    if arcpy.Exists(veg_eraseBurn):
        arcpy.Delete_management(veg_eraseBurn)

    print 'Erasing areas of high burn severity from the dissolved 4D, 5D, 6D, 4M, 5M, and 6M vegetation type feature class...'
    arcpy.Erase_analysis(veg_dissolve, kingHigh_layer, veg_eraseBurn, "")

    print 'Created veg layer reflecting the area lost to high severity burn in the 2014 King Fire.'

    # Process: Add Geometry Attributes        
    # Calculating acreage of veg parcels from dissolve step above

    print 'Calculating area of dissolved vegetation parcels without high burn severity...'

    arcpy.AddGeometryAttributes_management(veg_eraseBurn, "AREA", "", "ACRES", "")

    # Process: Select Layer By Attribute - Selecting polygons with 10 acres or greater area
    # Checking for previous versions of layer
    print 'Checking for previous versions of vegetation layer with 10 or more acres of suitable habitat...'

    if arcpy.Exists(vegAcre_layer):
        arcpy.Delete_management(vegAcre_layer)
    
    # Creating selection of areas with 10 or more acres
    print 'Creating feature layer with parcels greater than or equal to 10 acres...'

    arcpy.MakeFeatureLayer_management(veg_eraseBurn, vegAcre_layer)

    # Select for areas from the veg layer that are > 10 acres
    print 'Selecting areas where there is greater than 10 acres of contiguous suitable habitat...'
    
    query3 = """"POLY_AREA" >= 10"""
    arcpy.SelectLayerByAttribute_management(vegAcre_layer, "NEW_SELECTION", query3)

    # Process: Buffer on Quidazoic timber units
    # Checking for previous versions of layer
    print 'Checking for previous versions of buffered Quidazoic units...'

    if arcpy.Exists(quid_buffer):
        arcpy.Delete_management(quid_buffer)
    
    print 'Creating quarter mile buffer on Quidazoic timber sale units...'
    arcpy.Buffer_analysis(Quidazoic_Units, quid_buffer, "0.25 Miles", "FULL", "ROUND", "ALL", "", "GEODESIC")

    # Process: Select Layer By Location - Goshawk PACs within Quidazoic timber sale units
    # Checking for previous versions of layer
    print 'Checking for previous versions of goshawk PACs within Quidazoic timber sale...'

    if arcpy.Exists(nogoQuid_layer):
        arcpy.Delete_management(nogoQuid_layer)

    # Creating selection of goshawk PACs within Quidazoic timber sale
    print 'Creating feature layer with goshawk PACs...'

    arcpy.MakeFeatureLayer_management(NOGOpacs, nogoQuid_layer)
    
    # Select for those goshawk PACs that fall within the project area, in this case, Quidazoic Timber Sale
    print 'Selecting goshawk PACs that fall within the Quidazoic timber sale area...'
    
    arcpy.SelectLayerByLocation_management(nogoQuid_layer, "INTERSECT", Quidazoic_Units, "", "NEW_SELECTION", "NOT_INVERT")

    # Process: Buffer on Goshawk PACs
    # Checking for previous versions of layer
    print 'Checking for previous versions of buffered goshawk PACs...'

    if arcpy.Exists(nogo_buffer):
        arcpy.Delete_management(nogo_buffer)
    
    print 'Creating quarter mile buffer on Quidazoic timber sale unitsgoshawk PACs in project area...'
    
    arcpy.Buffer_analysis(nogoQuid_layer, nogo_buffer, "0.25 Miles", "FULL", "ROUND", "ALL", "", "GEODESIC")

    # Process: Merge NOGO PACs with buffer and project layer
    # Checking for previous versions 

    print 'Checking for previous versions of merged goshawk PACs and project area with vegetation features...'

    if arcpy.Exists(vegNogo):
        arcpy.Delete_management(vegNogo)

    print 'Combining PACs with project area to determine survey area within Quidazoic...'
    arcpy.Merge_management([quid_buffer, nogo_buffer], vegNogo)

    print 'File created reflecting survey area within PACs and project area.'

    # Process: Clip veg to Quidazoic timber sale units and goshawk buffers
    # Checking for previous versions 

    print 'Checking for previous versions of clipped vegetation feature class...'

    if arcpy.Exists(veg_nogoClip):
        arcpy.Delete_management(veg_nogoClip)

    print 'Clipping vegetation to Quidazoic survey area...'
    arcpy.Clip_analysis(vegAcre_layer, vegNogo, veg_nogoClip, "")


    # Process: Select Layer By Attribute - Ensuring areas being surveyed are not on private property
    # Checking for previous versions of layer
    print 'Checking for previous versions of private property layer...'

    if arcpy.Exists(private_layer):
        arcpy.Delete_management(private_layer)

    # Creating selection of private property within Eldorado National Forest
    print 'Creating feature layer with private property...'

    arcpy.MakeFeatureLayer_management(ownership, private_layer)

    query3 = """"OWNERCLASS" = 'NON-FS'"""
  
    arcpy.SelectLayerByAttribute_management(private_layer, "NEW_SELECTION", query3)

    # Process: Erasing private property from combined veg and NOGO data
    # Checking for previous versions 

    print 'Checking for previous versions of survey area with private property removed...'

    if arcpy.Exists(vegNogo_erasePrivate):
        arcpy.Delete_management(vegNogo_erasePrivate)

    arcpy.Erase_analysis(veg_nogoClip, private_layer, vegNogo_erasePrivate, "")

    # Process: Select Layer By Attribute - Removing areas of slope greater than 35%
    # Checking for previous versions of layer
    print 'Checking for previous versions of layer containing slope greater than 35%...'

    if arcpy.Exists(slope_layer):
        arcpy.Delete_management(slope_layer)

    # Creating selection of slope greater than 35%
    print 'Creating feature layer reflecting slope greater than 35%...'

    arcpy.MakeFeatureLayer_management(Slope35, slope_layer)

    query4 = """"SlopePercent" = '>35%'"""
    
    arcpy.SelectLayerByAttribute_management(slope_layer, "NEW_SELECTION", query4)

    # Process: Erase - erasing slope from  previous vegNogo layer
    # Checking for previous versions of file
    print 'Checking for previous versions of layer containing slope greater than 35%...'

    if arcpy.Exists(vegNogo_eraseSlope):
        arcpy.Delete_management(vegNogo_eraseSlope)

    print 'Erasing slope from survey area...'
    
    arcpy.Erase_analysis(vegNogo_erasePrivate, slope_layer, vegNogo_eraseSlope, "")

    print 'Survey area created. Points can be added using Make Sample Point Grid toolbox.'

    #Ended up not being able to incorporate this part of the script, so substituted fishnet tool above
    #Process: Make Sample Point Grid
    #arcpy.gp.toolbox = "C:/PythonPrimer/FinalProject/MakeSamplePointGrid.tbx";
    #Warning: the toolbox C:/PythonPrimer/FinalProject/MakeSamplePointGrid.tbx DOES NOT have an alias. 
    #Please assign an alias to avoid tool name collisions for tbx
    #And replace arcpy.gp.MakeSamplePointGrid(...) with arcpy.MakeSamplePointGrid_ALIAS(...)
    #the following line is the original code from model builder:
    #arcpy.gp.MakeSamplePointGrid(vegNogo_eraseSlope, "Interval", "10", "200", "100", Offset_alternating_rows, NOGO_surveyLocs)
    #arcpy.MakeSamplePointGrid_SamplePointGrid(vegNogo_eraseSlope, "Interval", "10", "200", "100", Offset_alternating_rows, NOGO_surveyLocs)


except:
    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print msgs
    print pymsg
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print arcpy.GetMessages(1)



