#Final Project
#Created by: Tammy Dong
#Date created: 5/1/15

#Improving Georeferencing processes
#This script will help automate processing that takes place in georeferencing data
#It is tailored to the data from the Consortium of California Herbaria.
#Consortium website: http://ucjeps.berkeley.edu/consortium/

#import modules

import arcpy, sys, os, traceback, fnmatch


#datapath

datapath = "C:\\Users\\Rachel\\Desktop\\PYTHONCLASS\\final_project\\data\\"

#path to folder where intermediate tables go after separating by datum
int_datum_table_folder = os.path.join(datapath, "Int_datum_table")

#path to manual folders
#where tables that need to be sorted through row by row goes

man_folder = os.path.join(datapath, "manual")

#where tables go for display
dis_folder = os.path.join(datapath, "Displaytables")

#shapefile folders

shp_folder = os.path.join(datapath, "Shapefiles")

#for shapefiles that are projected
TA_folder = os.path.join(shp_folder, "TealeAlbers")

#for shapefiles that are unprojected but have defined coordinate systems 
OGCS_folder = os.path.join(shp_folder, "originalGCS")

#default table name from website is "get_consort"
consort_table = os.path.join(datapath,"get_consort.dbf")


try:
    #delete files in intermediate folders just in case there are any extraneous files
    #from previous "get_consort" or else it may become merged at the end
    #for example if previous "get_consort" table had data with NAD27 but current does not,
    #it will not be deleted and will be merged at the end with current table data which is not desirable.
    arcpy.env.workspace = TA_folder
    shpfiles = arcpy.ListFeatureClasses()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = OGCS_folder
    shpfiles = arcpy.ListFeatureClasses()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = dis_folder
    shpfiles = arcpy.ListTables()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = man_folder
    shpfiles = arcpy.ListTables()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = int_datum_table_folder
    shpfiles = arcpy.ListTables()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    #1. Separate data based on datum.

    print "Creating new tables based on datum..."
    
    #variables:

    tableview = "consort_view"
    intable = consort_table
    
    #delete existing table view

    if arcpy.Exists (tableview):
        arcpy.Delete_management (tableview)

    #make table view
    
    arcpy.MakeTableView_management(intable, tableview)

    #Use search cursor to access table
    #create loop to select and export different datum types using copy rows

    field_name = ["datum"]
    with arcpy.da.SearchCursor(tableview, field_name) as Drows:
        for Drow in Drows:
            Datum_name = Drow[0]
            query = """"datum" = '"""+ Datum_name +"""'"""

            arcpy.SelectLayerByAttribute_management (tableview, "CLEAR_SELECTION")
        
            arcpy.SelectLayerByAttribute_management (tableview, "NEW_SELECTION", query)

            #if then statement to save unknown datum to the manual folder.
            #unknown datum is represented by a space in attribute table
            
            if Datum_name == " ":
                out_table = os.path.join(man_folder, 'Unknown_consort.dbf')

            #else statement puts other tables in intermediate table folder.
            else:
                out_table = os.path.join(int_datum_table_folder, Datum_name + '_consort.dbf')

            if arcpy.Exists (out_table):
                arcpy.Delete_management (out_table)

            arcpy.CopyRows_management (tableview, out_table)

            arcpy.SelectLayerByAttribute_management (tableview, "CLEAR_SELECTION")

        del Drow, Drows
    del tableview  

    #list tables in intermediate datum and manual folder.
    #reference: http://resources.arcgis.com/en/help/main/10.1/index.html#//018v0000002t000000

    print "Located in " + man_folder + ":"
    
    arcpy.env.workspace = man_folder

    tables = arcpy.ListTables()
    for table in tables:
        print(table)

    print "Located in " + int_datum_table_folder + ":"
    
    arcpy.env.workspace = int_datum_table_folder

    tables = arcpy.ListTables()
    for table in tables:
        print(table)

    print "Creating new tables based on coordinate precision and adding buffer field..."
    
    for table in tables:
        #2. separate intermediate tables for potential truncated coordinates
        #truncated coordinates defined by less than 4 decimal places
        #create new field for lat/long as text data instead or float or double because Query will count characters
        new_field1 = "lattext"
        new_field2 = "longtext"
        #add another field for the error radius buffer later
        #the error radius and units are in two different fields
        new_field3 = "buffer_fld"

        arcpy.AddField_management (table, new_field1, "TEXT")
        arcpy.AddField_management (table, new_field2, "TEXT")
        arcpy.AddField_management (table, new_field3, "TEXT")

        #update cursor to fill out attributes for new fields
        #ref: http://resources.arcgis.com/en/help/main/10.2/index.html#//03q30000008n000000
        rows = arcpy.UpdateCursor(table)
        for row in rows:
            row.setValue(new_field1, row.getValue("latitude"))
            row.setValue(new_field2, row.getValue("longitude"))
            row.setValue(new_field3, str(row.getValue("error_dist"))+ " " +row.getValue("units"))

            rows.updateRow(row)

        #using search cursor and select by attribute, export data to manual and display folders
        if arcpy.Exists ("tblv"):
            arcpy.Delete_management ("tblv")

        arcpy.MakeTableView_management(table, "tblv")

        with arcpy.da.SearchCursor("tblv", [new_field1]) as Drows:
            for Drow in Drows:

                #query counts characters to the right of decimal point for latfive and longfive
                #note, this query will not work with numbers
                
                query = """"lattext" LIKE '%.___' Or "lattext" LIKE '%.__' Or "lattext" LIKE '%._' Or "lattext" LIKE '__' Or "longtext" LIKE '%.___' Or "longtext" LIKE '%.__' Or "longtext" LIKE '%._' Or "longtext" LIKE '____'"""

                arcpy.SelectLayerByAttribute_management ("tblv", "CLEAR_SELECTION")
        
                arcpy.SelectLayerByAttribute_management ("tblv", "NEW_SELECTION", query)

                out_truntable = os.path.join(man_folder, "trun" + table)

                if arcpy.Exists (out_truntable):
                    arcpy.Delete_management (out_truntable)

                arcpy.CopyRows_management ("tblv", out_truntable)

                out_distable = os.path.join(dis_folder, table)

                if arcpy.Exists (out_distable):
                    arcpy.Delete_management (out_distable)
                    
                # switch selection from previous query to select nontruncated lat/long values
                arcpy.SelectLayerByAttribute_management ("tblv", "SWITCH_SELECTION")

                arcpy.CopyRows_management ("tblv", out_distable)

                arcpy.SelectLayerByAttribute_management ("tblv", "CLEAR_SELECTION")       

        del rows, row, Drows, Drow

    #list tables in display and manual folder.
    #reference: http://resources.arcgis.com/en/help/main/10.1/index.html#//018v0000002t000000

    print "Located in " + man_folder + ":"
    
    arcpy.env.workspace = man_folder

    tables = arcpy.ListTables()
    for table in tables:
        print(table)

    print "Located in " + dis_folder + ":"
    
    arcpy.env.workspace = dis_folder

    tables = arcpy.ListTables()
    for table in tables:
        print(table)

    #3.Display x,y coordinates and create point shapefile
    #first create loop to make table view of all tables in dis_folder and then use MakeXYEventLayer_management tool (uses table view as input).
    print "Creating shapefiles from tables..."
    
    for table in tables:

        if arcpy.Exists ("tblv"):
            arcpy.Delete_management ("tblv")

        arcpy.MakeTableView_management(table, "tblv")

        #if elif statements to individualize coordinate system
        #use arcpy.spatialreference to set spatial reference
        #http://resources.arcgis.com/en/help/main/10.1/index.html#//018z0000000v000000
        
        #using fnmatch.fnmatch to match tables filename key words to spatial reference.
        #https://docs.python.org/2/library/fnmatch.html

        if fnmatch.fnmatch(table, '*84*.dbf'):
            spatial_ref = arcpy.SpatialReference("WGS 1984")
            

        elif fnmatch.fnmatch(table, '*27*.dbf'):
            spatial_ref = arcpy.SpatialReference("NAD 1927")
            

        elif fnmatch.fnmatch(table, '*83*.dbf'):
            spatial_ref = arcpy.SpatialReference("NAD 1983")
            
        else:
            print "spatial reference issues"

        #referenece: http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//00170000006z000000

        XYEventoutlayer = table + "_pts"

        #this makes a temporary point file
        if arcpy.Exists (XYEventoutlayer):
            arcpy.Delete_management (XYEventoutlayer)

        arcpy.MakeXYEventLayer_management ("tblv", "longitude", "latitude",  XYEventoutlayer, spatial_ref)

        save_event_layer = os.path.join(OGCS_folder, table + ".shp")

        if arcpy.Exists (save_event_layer):
            arcpy.Delete_management (save_event_layer)

        #save XY event to shapefile using copy.features
        
        arcpy.CopyFeatures_management(XYEventoutlayer, save_event_layer)
            
        
    #list out shapefiles in shapefile folder using arcpy.ListFeatureClasses
    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//000v0000001n000000
    print "Located in " + OGCS_folder + ":"
    
    arcpy.env.workspace = OGCS_folder
    
    shpfiles = arcpy.ListFeatureClasses()
    for shpfile in shpfiles:
        print(shpfile)

    #4. project shapefiles to teale albers 
    print "Projecting shapefiles to Teale Alberts..."

    for shpfile in shpfiles:
        #project shapefiles to teale alberts
        

        #first need to define spatial references and transformation

        if fnmatch.fnmatch(shpfile, '*84*.shp'):
            spatial_ref = arcpy.SpatialReference("NAD 1983 California (Teale) Albers (Meters)")
            transformation = "NAD_1983_To_WGS_1984_5"
            
        elif fnmatch.fnmatch(shpfile, '*27*.shp'):
            spatial_ref = arcpy.SpatialReference("NAD 1983 California (Teale) Albers (Meters)")
            transformation = "NAD_1927_To_NAD_1983_NADCON"

        elif fnmatch.fnmatch(shpfile, '*83*.shp'):
            spatial_ref = arcpy.SpatialReference("NAD 1983 California (Teale) Albers (Meters)")
            transformation = "''"

        else:
            print "spatial reference issues"

        outshpfile = os.path.join(TA_folder, "TA" + shpfile)

        if arcpy.Exists (outshpfile):
            arcpy.Delete_management (outshpfile)
        
        arcpy.Project_management (shpfile, outshpfile, spatial_ref, transformation)

    #List shapefiles that were just created
    print "Located in " + TA_folder + ":"
    arcpy.env.workspace = TA_folder
    
    shpfiles = arcpy.ListFeatureClasses()
    for shpfile in shpfiles:
        print(shpfile)

    #5. merge shapefiles that were just created
    #http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//001700000055000000
    print "merging shapefiles..."
    merge_output = "TA_pts_merge.shp"

    if arcpy.Exists (merge_output):
            arcpy.Delete_management (merge_output)
            
    arcpy.Merge_management (shpfiles, merge_output)

    #6. Create error radius via buffering merged shapefile using buffer field created previously 
    #http://resources.arcgis.com/en/help/main/10.1/index.html#//000800000019000000
    print "buffering merged shapefile points..."
    buffer_output = "TA_pts_error.shp"
    
    if arcpy.Exists (buffer_output):
            arcpy.Delete_management (buffer_output)

    arcpy.Buffer_analysis(merge_output, buffer_output, new_field3)

    print "get_consort table processing completed"

    #optional: delete intermediate data that are not needed. 

    '''
    arcpy.env.workspace = OGCS_folder
    shpfiles = arcpy.ListFeatureClasses()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = dis_folder
    shpfiles = arcpy.ListTables()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    arcpy.env.workspace = int_datum_table_folder
    shpfiles = arcpy.ListTables()
    for shpfile in shpfiles:
        if arcpy.Exists (shpfile):
            arcpy.Delete_management (shpfile)

    '''

except:
    # http://help.arcgis.com/en/arcgisdesktop/10.0/help/index.html#//002z0000000q000000

    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n     " +        str(sys.exc_type) + ": " + str(sys.exc_value) + "\n"
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print msgs
    print pymsg
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print arcpy.GetMessages(1)

